/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/hypergeometric/cdf' );
import Hypergeometric = require( './../../../../../base/dists/hypergeometric/ctor' );
import kurtosis = require( './../../../../../base/dists/hypergeometric/kurtosis' );
import logpmf = require( './../../../../../base/dists/hypergeometric/logpmf' );
import mean = require( './../../../../../base/dists/hypergeometric/mean' );
import mode = require( './../../../../../base/dists/hypergeometric/mode' );
import pmf = require( './../../../../../base/dists/hypergeometric/pmf' );
import quantile = require( './../../../../../base/dists/hypergeometric/quantile' );
import skewness = require( './../../../../../base/dists/hypergeometric/skewness' );
import stdev = require( './../../../../../base/dists/hypergeometric/stdev' );
import variance = require( './../../../../../base/dists/hypergeometric/variance' );

/**
* Interface describing the `hypergeometric` namespace.
*/
interface Namespace {
	/**
	* Hypergeometric distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param N - population size
	* @param K - subpopulation size
	* @param n - number of draws
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 1.0, 8, 4, 2 );
	* // returns ~0.786
	*
	* y = ns.cdf( 1.5, 8, 4, 2 );
	* // returns ~0.786
	*
	* y = ns.cdf( 2.0, 8, 4, 2 );
	* // returns 1.0
	*
	* y = ns.cdf( 0.0, 8, 4, 2 );
	* // returns ~0.214
	*
	* var mycdf = ns.cdf.factory( 30, 20, 5 );
	* y = mycdf( 4.0 );
	* // returns ~0.891
	*
	* y = mycdf( 1.0 );
	* // returns ~0.031
	*/
	cdf: typeof cdf;

	/**
	* Hypergeometric Distribution.
	*/
	Hypergeometric: typeof Hypergeometric;

	/**
	* Returns the excess kurtosis of a hypergeometric distribution.
	*
	* ## Notes
	*
	* -   If provided a population size `N`, subpopulation size `K` or draws `n` which is not a nonnegative integer, the function returns `NaN`.
	* -   If the number of draws `n` or subpopulation size `K` exceed population size `N`, the function returns `NaN`.
	*
	* @param N - population size
	* @param K - subpopulation size
	* @param n - number of draws
	* @returns kurtosis
	*
	* @example
	* var v = ns.kurtosis( 16, 11, 4 );
	* // returns ~-0.326
	*
	* @example
	* var v = ns.kurtosis( 4, 2, 2 );
	* // returns 0.0
	*
	* @example
	* var v = ns.kurtosis( 10, 5, 12 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 10.3, 10, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 10, 5.5, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 10, 5, 4.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( NaN, 10, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 20, NaN, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 20, 10, NaN );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Hypergeometric distribution natural logarithm of the probability mass function (logPMF).
	*
	* @param x - input value
	* @param N - population size
	* @param K - subpopulation size
	* @param n - number of draws
	* @returns evaluated logPMF
	*
	* @example
	* var y = ns.logpmf( 1.0, 8, 4, 2 );
	* // returns ~-0.56
	*
	* y = ns.logpmf( 2.0, 8, 4, 2 );
	* // returns ~-1.54
	*
	* y = ns.logpmf( 0.0, 8, 4, 2 );
	* // returns ~-1.54
	*
	* y = ns.logpmf( 1.5, 8, 4, 2 );
	* // returns -Infinity
	*
	* var mylogpmf = ns.logpmf.factory( 30, 20, 5 );
	* y = mylogpmf( 4.0 );
	* // returns ~-1.079
	*
	* y = mylogpmf( 1.0 );
	* // returns ~-3.54
	*/
	logpmf: typeof logpmf;

	/**
	* Returns the expected value of a hypergeometric distribution.
	*
	* ## Notes
	*
	* -   If provided a population size `N`, subpopulation size `K` or draws `n` which is not a nonnegative integer, the function returns `NaN`.
	* -   If the number of draws `n` or subpopulation size `K` exceed population size `N`, the function returns `NaN`.
	*
	* @param N - population size
	* @param K - subpopulation size
	* @param n - number of draws
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 16, 11, 4 );
	* // returns 2.75
	*
	* @example
	* var v = ns.mean( 2, 1, 1 );
	* // returns 0.5
	*
	* @example
	* var v = ns.mean( 10, 5, 12 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 10.3, 10, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 10, 5.5, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 10, 5, 4.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( NaN, 10, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 20, NaN, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 20, 10, NaN );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the mode of a hypergeometric distribution.
	*
	* ## Notes
	*
	* -   If provided a population size `N`, subpopulation size `K` or draws `n` which is not a nonnegative integer, the function returns `NaN`.
	* -   If the number of draws `n` or subpopulation size `K` exceed population size `N`, the function returns `NaN`.
	*
	* @param N - population size
	* @param K - subpopulation size
	* @param n - number of draws
	* @returns mode
	*
	* @example
	* var v = ns.mode( 16, 11, 4 );
	* // returns 3
	*
	* @example
	* var v = ns.mode( 2, 1, 1 );
	* // returns 1
	*
	* @example
	* var v = ns.mode( 10, 5, 12 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 10.3, 10, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 10, 5.5, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 10, 5, 4.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( NaN, 10, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 20, NaN, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 20, 10, NaN );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Hypergeometric distribution probability mass function (PMF).
	*
	* @param x - input value
	* @param N - population size
	* @param K - subpopulation size
	* @param n - number of draws
	* @returns evaluated PMF
	*
	* @example
	* var y = ns.pmf( 5.0, 20.0, 0.8 );
	* // returns ~0.157
	*
	* y = ns.pmf( 21.0, 20.0, 0.5 );
	* // returns ~0.06
	*
	* y = ns.pmf( 5.0, 10.0, 0.4 );
	* // returns ~0.016
	*
	* y = ns.pmf( 0.0, 10.0, 0.9 );
	* // returns ~0.349
	*
	* y = ns.pmf( 21.0, 15.5, 0.5 );
	* // returns ~0.037
	*
	* y = ns.pmf( 5.0, 7.4, 0.4 );
	* // returns ~0.051
	*
	* var mypmf = ns.pmf.factory( 10, 0.5 );
	* y = mypmf( 3.0 );
	* // returns ~0.027
	*
	* y = mypmf( 5.0 );
	* // returns ~0.061
	*/
	pmf: typeof pmf;

	/**
	* Hypergeometric distribution quantile function.
	*
	* @param p - input value
	* @param N - population size
	* @param K - subpopulation size
	* @param n - number of draws
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.4, 40, 20, 10 );
	* // returns 5
	*
	* y = ns.quantile( 0.8, 60, 40, 20 );
	* // returns 15
	*
	* y = ns.quantile( 0.5, 100, 10, 10 );
	* // returns 1
	*
	* y = ns.quantile( 0.0, 100, 40, 20 );
	* // returns 0
	*
	* y = ns.quantile( 1.0, 100, 40, 20 );
	* // returns 20
	*
	* var myquantile = ns.quantile.factory( 100, 20, 10 );
	* y = myquantile( 0.2 );
	* // returns 1
	*
	* y = myquantile( 0.9 );
	* // returns 4
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness of a hypergeometric distribution.
	*
	* ## Notes
	*
	* -   If provided a population size `N`, subpopulation size `K` or draws `n` which is not a nonnegative integer, the function returns `NaN`.
	* -   If the number of draws `n` or subpopulation size `K` exceed population size `N`, the function returns `NaN`.
	*
	* @param N - population size
	* @param K - subpopulation size
	* @param n - number of draws
	* @returns skewness
	*
	* @example
	* var v = ns.skewness( 16, 11, 4 );
	* // returns ~-0.258
	*
	* @example
	* var v = ns.skewness( 4, 2, 2 );
	* // returns 0.0
	*
	* @example
	* var v = ns.skewness( 10, 5, 12 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 10.3, 10, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 10, 5.5, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 10, 5, 4.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( NaN, 10, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 20, NaN, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 20, 10, NaN );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation of a hypergeometric distribution.
	*
	* ## Notes
	*
	* -   If provided a population size `N`, subpopulation size `K` or draws `n` which is not a nonnegative integer, the function returns `NaN`.
	* -   If the number of draws `n` or subpopulation size `K` exceed population size `N`, the function returns `NaN`.
	*
	* @param N - population size
	* @param K - subpopulation size
	* @param n - number of draws
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 16, 11, 4 );
	* // returns ~0.829
	*
	* @example
	* var v = ns.stdev( 2, 1, 1 );
	* // returns 0.5
	*
	* @example
	* var v = ns.stdev( 10, 5, 12 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 10.3, 10, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 10, 5.5, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 10, 5, 4.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( NaN, 10, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 20, NaN, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 20, 10, NaN );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of a hypergeometric distribution.
	*
	* ## Notes
	*
	* -   If provided a population size `N`, subpopulation size `K` or draws `n` which is not a nonnegative integer, the function returns `NaN`.
	* -   If the number of draws `n` or subpopulation size `K` exceed population size `N`, the function returns `NaN`.
	*
	* @param N - population size
	* @param K - subpopulation size
	* @param n - number of draws
	* @returns variance
	*
	* @example
	* var v = ns.variance( 16, 11, 4 );
	* // returns ~0.688
	*
	* @example
	* var v = ns.variance( 2, 1, 1 );
	* // returns 0.25
	*
	* @example
	* var v = ns.variance( 10, 5, 12 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 10.3, 10, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 10, 5.5, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 10, 5, 4.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( NaN, 10, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 20, NaN, 4 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 20, 10, NaN );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Hypergeometric distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
