/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Clippings.
 *
 * The Initial Developer of the Original Code is 
 * Alex Eng <ateng@users.sourceforge.net>.
 * Portions created by the Initial Developer are Copyright (C) 2005-2008
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * ***** END LICENSE BLOCK ***** */

var gClippingsList, gStatusBar;
var gMoveToMnu1, gMoveToMnu2, gCopyToMnu1, gCopyToMnu2;
var gCurrentListItemIndex = -1;
var gStrBundle;
var gAppUtils;
var gDataSource;
var gClippingsListener;
var gIsFolderMenuSeparatorInitialized = {};
var gSaveInProgress = false;
var gIsClippingsDirty = false;
var gJustMigrated = false;
var gCurrentFile;


// Clippings XPCOM service
var gClippingsSvc;

// Undo
var gUndoStack = {
  length: 0,
  _stack: [],

  push: function (aState)
  {
    this._stack.push(aState);
    this.length++;
  },

  pop: function ()
  {
    var rv = this._stack.pop();
    this.length--;
    return rv;
  }
};

// Redo - only 1 undo action is reversible
var gRedoStack = {
  length:   0,
  _lastUndo: null,

  push: function (aState)
  {
    this._lastUndo = aState;
    this.length = (this.length == 0 ? 1 : 1);
  },

  pop: function ()
  {
    var rv = {};
    for (var ppty in this._lastUndo) {
      rv[ppty] = this._lastUndo[ppty];
    }
    this._lastUndo = null;
    this.length = 0;
    return rv;
  }
};


// Flags for gUndoStack._stack[i].action
const ACTION_EDITNAME = 1;
const ACTION_EDITTEXT = 2;
const ACTION_DELETECLIPPING = 3;
const ACTION_CREATENEW = 4;
const ACTION_CHANGEPOSITION = 5;
const ACTION_CREATENEWFOLDER = 6;
const ACTION_DELETEFOLDER = 7;
const ACTION_MOVETOFOLDER = 8;
const ACTION_COPYTOFOLDER = 9;
const ACTION_DELETEEMPTYFOLDER = 10;
const ACTION_SETSHORTCUTKEY = 11;

// Flags for aDestUndoStack parameter of functions for reversible actions
const UNDO_STACK = 1;
const REDO_STACK = 2;


//
// DOM utility function
//

function $(aID)
{
  return document.getElementById(aID);
}


//
// Drag 'n drop handlers for Clippings Manager's tree list
//

var dndStartPos = null;
var dndStartURI = null;
var dndExtText = null;

var listitemObserver = {
  onDragStart: function (event, transferData, action)
  {
    if (event.target.tagName != "treechildren") {
      return;
    }
    var index = gClippingsList.tree.boxObject.getRowAt(event.clientX,
						       event.clientY);
    var uri = gClippingsList.getURIAtIndex(index);
    var pos = gClippingsSvc.ctrIndexOf(uri);
    dndStartURI = uri;
    dndStartPos = pos;

    transferData.data = new TransferData();
    transferData.data.addDataForFlavour("text/unicode", pos);
  },

  onDragOver:  function (event, flavour, session) {},
  onDragExit:  function (event, session) {},

  onDrop: function (event, transferData, session) 
  {
    if (transferData.data != "") {
      dndExtText = transferData.data;
    }
  },

  getSupportedFlavours: function () 
  {
    var flavours = new FlavourSet();
    flavours.appendFlavour("text/unicode");
    return flavours;
  } 
};


var treeBuilderObserver = {
  // Mozilla 1.7 (Firefox 1.0) only
  canDropBeforeAfter: function (idx, orient)
  {
    return true;
  },

  // Mozilla 1.7 (Firefox 1.0) only
  canDropOn: function (idx, orient)
  {
    return true;
  },

  canDrop: function (idx, orient)
  {
    // Return true to allow dropping into a folder.  Return the `orient' param
    // to allow dropping only if the folder is expanded in the tree list.
    return true;
  },

  onDrop: function (idx, orient)
  {
    var uri = gClippingsList.getURIAtIndex(idx);
    var newPos = gClippingsSvc.ctrIndexOf(uri);
    // `orient' is 0 (zero) if dragging and dropping into a folder item.
    var destParent = orient == 0 ? uri : gClippingsSvc.getParent(uri);

    gAppUtils.log("DnD orientation: " + orient + "\nnewPos: " + newPos);

    if (! dndStartURI) {
      // Creating a new clipping from text that is dragged and dropped into
      // Clippings Manager from an external app.
      if (dndExtText) {
	if (orient == 1) {  // nsIXULTreeBuilderObserver.DROP_AFTER
	  newPos++;
	}
	else if (orient == 0) {  // nsIXULTreeBuilderObserver.DROP_ON
	  newPos = null;  // Will append to end of folder being dropped to.
	}
	var name = gClippingsSvc.createClippingNameFromText(dndExtText, 0);
	var text = dndExtText;
	gClippingsSvc.createNewClippingEx(destParent, null, name, text, newPos, false);
      }
      else {
	gAppUtils.beep();
	gAppUtils.log("Unknown item dropped into tree list; ignoring.");
      }

      gIsClippingsDirty = true;
      this._endDnD();
      return;
    }

    var srcParent = gClippingsSvc.getParent(dndStartURI);

    if (srcParent == destParent) {
      if (orient == 0) {  // nsIXULTreeBuilderObserver.DROP_ON
	// Dragging and dropping a folder item into its containing folder.
	gAppUtils.beep();
	gAppUtils.log("The selected item already belongs in this folder!");
	this._endDnD();
	return;
      }
      else if (orient == -1) { // nsIXULTreeBuilderObserver.DROP_BEFORE
	if (dndStartPos < newPos) newPos--;
      }
      else if (orient == 1) {  // nsIXULTreeBuilderObserver.DROP_AFTER
	if (dndStartPos > newPos) newPos++;
      }

      gAppUtils.log(String.ae_fmt("Orientation: %d\nChanging position from %d to %d", orient, dndStartPos, newPos));

      if (dndStartPos != newPos) {
	moveEntry(dndStartURI, srcParent, dndStartPos, newPos, UNDO_STACK);
	gClippingsList.selectedURI = dndStartURI;
      }
    }
    else {
      if (dndStartURI == destParent) {
	gAppUtils.beep();
	gAppUtils.log("Cannot move a folder into itself!");
	this._endDnD();
	return;
      }

      // Prevent infinite recursion due to moving a folder into its own subfolder.
      if (gClippingsSvc.isFolder(dndStartURI)) {
	var parentURI = gClippingsSvc.getParent(destParent);
	while (parentURI && parentURI != gClippingsSvc.kRootFolderURI) {
	  if (parentURI == dndStartURI) {
	    gAppUtils.beep();
	    gAppUtils.log("Cannot move a folder into a subfolder of itself!");
	    this._endDnD();
	    return;
	  }
	  parentURI = gClippingsSvc.getParent(parentURI);
	}
      }

      if (orient == 1) {
	// nsIXULTreeBuilderObserver.DROP_AFTER
	newPos++;
      }
      else if (orient == 0) {
	// Dragging and dropping into a folder - append the item to be
	// moved as the last item of the folder.
	newPos = null;
      }

      gAppUtils.log(String.ae_fmt("Orientation: %d\nMoving to folder %S at position %d", orient, gClippingsSvc.getName(destParent), newPos));

      moveToFolderHelper(dndStartURI, srcParent, destParent, null, newPos,
			 UNDO_STACK, true);
    }

    updateDisplay();
    this._endDnD();
  },

  _endDnD: function ()
  {
    dndStartPos = null;
    dndStartURI = null;
    dndExtText = null;
  },
  
  onSelectionChanged: function () {},
  onToggleOpenState: function (idx) {}
};


//
// QuickEdit - instant updating of clipping text edits, without the need
// for a "Save" command to be manually invoked
//

var gQuickEdit = {
  _oldValue:  "",
  _tid:       null,
  _interval:  1000,

  init: function (aOldValue)  {
    this._oldValue = aOldValue;
    this._interval = gAppUtils.getPref("clippings.clipmgr.quickedit.update_interval");
  },
  
  start: function () {
    this._tid = window.setInterval("gQuickEdit._updateText()", this._interval);
  },
  
  isStarted: function () {
    return (this._tid != null);
  },

  stop: function () 
  {
    if (this._tid != null) {
      window.clearInterval(this._tid);
      gQuickEdit._tid = null;
      gQuickEdit._oldValue = "";
    }
  },

  _updateText: function () 
  {
    if (!gClippingsSvc || !gClippingsList || gCurrentListItemIndex == -1) {
      this.stop();
      return;
    }

    var currentURI = gClippingsList.selectedURI;
    if (!currentURI || !gClippingsSvc.exists(currentURI) 
	|| !gClippingsSvc.isClipping(currentURI)) {
      this.stop();
      return;
    }

    var clippingTextElt = document.getElementById("clipping-text");
    var newText = clippingTextElt.value;

    if (newText != this._oldValue) {
      updateText(newText);
      this._oldValue = "";
    }
  }
};


//
// Shortcut key editing
//

var gShortcutKey = {
  _oldKey:   "",
  _oldIndex: -1,
  
  setOldKey: function ()
  {
    if (gCurrentListItemIndex == -1) {
      return;
    }
    
    var clippingKey = document.getElementById("clipping-key");
    var uri = gClippingsList.getURIAtIndex(gCurrentListItemIndex);
    this._oldKey = gClippingsSvc.getShortcutKey(uri);
    this._oldIndex = clippingKey.selectedIndex;
  },

  update: function (aDestUndoStack)
  {
    if (gCurrentListItemIndex == -1) {
      return;
    }

    var key = "";
    var uri = gClippingsList.getURIAtIndex(gCurrentListItemIndex);
    var clippingKey = document.getElementById("clipping-key");

    if (clippingKey.selectedIndex == 0) {
      if (! this._oldKey) {
	// Skip shortcut key update if none was ever defined.
	return;
      }
    }
    else {
      key = clippingKey.menupopup.childNodes[clippingKey.selectedIndex].label;
    }

    if (key == this._oldKey) {
      return;
    }

    var keyDict = gClippingsSvc.getShortcutKeyDict();

    if (keyDict.hasKey(key)) {
      gAppUtils.beep();
      doAlert(gStrBundle.getString("errorShortcutKeyDefined"));
      clippingKey.selectedIndex = this._oldIndex;
      return;
    }

    gClippingsSvc.setShortcutKey(uri, key);
    gIsClippingsDirty = true;

    var state = { 
      action:  ACTION_SETSHORTCUTKEY, 
      uri:     uri, 
      key:     this._oldKey,
      keyIdx:  this._oldIndex
    };

    if (aDestUndoStack == UNDO_STACK) {
      gUndoStack.push(state);
    }
    else if (aDestUndoStack == REDO_STACK) {
      gRedoStack.push(state);
    }
    commit();
  }
};


//
// Common dialogs
//

function doAlert(aMessage)
{
  var title = gStrBundle.getString('appName');
  gAppUtils.alertEx(title, aMessage);
}


function doConfirm(aMessage)
{
  var rv;
  var title = gStrBundle.getString('appName');
  rv = gAppUtils.confirmEx(title, aMessage);
  return rv;
}



//
// Clippings Manager functions
//

function init() 
{
  gAppUtils = new aeClippingsUtils();

  try {
    gClippingsSvc = Components.classes["clippings@mozdev.org/clippings;1"]
                              .createInstance(Components.interfaces
					                .nsIClippingsService);
  }
  catch (e) {
    doAlert(e);
  }

  var treeElt = $("clippings-list");
  gClippingsList = new RDFTreeWrapper(treeElt);
  gClippingsList.tree.builder.addObserver(treeBuilderObserver);
  gStrBundle = $("ae-clippings-strings");
  gStatusBar = $("app-status");

  // HACK! Change the width of the outer <vbox>, not its inner <tree>. Invoking
  // window.getComputedStyle() on the tree element when unloading this window
  // will always return its width which is 4 pixels less than it actually is,
  // resulting in a gradually narrowing tree in successive sessions!
  var treeBox = $("tree-box");
  treeBox.style.width = gAppUtils.getPref("clippings.clipmgr.treelist.width");

  gAppUtils.log("Location of user profile directory: " + gAppUtils.getCurrentProfileURL());

  // Clippings backup
  var backupDirURL = gAppUtils.getDataSourcePathURL() + gAppUtils.BACKUP_DIR_NAME;
  gClippingsSvc.setBackupDir(backupDirURL);
  gClippingsSvc.setMaxBackupFiles(gAppUtils.getPref('clippings.backup.maxfiles'));

  window.setTimeout("clipeditStartup()", 1);
}


function clipeditStartup()
{
  var dlgArgs = { userCancel: null };
  var openSuccessful = false;
  var legacyDataSrcURL;

  do {
    window.openDialog("chrome://clippings/content/welcome.xul", "dlgWelcome", "chrome,dialog,modal,centerscreen", dlgArgs);

    if (dlgArgs.userCancel || ! dlgArgs.dataSrcURL) {
      window.close();
      return;
    }

    var dsURL = dlgArgs.dataSrcURL;
 
    if (dlgArgs.fileExists) {
      try {
	var fmt = validateFileFormat(dsURL);
      }
      catch (e if e.result == Components.results.NS_ERROR_FILE_NOT_FOUND) {
	doAlert(String.ae_fmt("%s: %s", gStrBundle.getString("errorFileNotFound"), dsURL));
	continue;
      }
      catch (e if e.result == Components.results.NS_ERROR_FILE_READ_ONLY) {
	doAlert(String.ae_fmt("%s: %s", gStrBundle.getString("errorFileReadOnly"), dsURL));
	continue;
      }
      catch (e if e.result == Components.results.NS_ERROR_OUT_OF_MEMORY) {
	doAlert(gStrBundle.getString("errorOutOfMemory"));
	continue;
      }
      catch (e if e.result == Components.results.NS_ERROR_FILE_ACCESS_DENIED) {
	doAlert(String.ae_fmt("%s: %s", gStrBundle.getString("errorAccessDenied"),
			      dsURL));
	continue;
      }
      catch (e if e.result == Components.results.NS_ERROR_FILE_IS_LOCKED) {
	doAlert(String.ae_fmt("%s: %s", gStrBundle.getString("errorFileLocked"),
			      dsURL));
	continue;
      }
      catch (e if e.result == Components.results.NS_ERROR_FILE_TOO_BIG) {
	doAlert(String.ae_fmt("%s: %s", gStrBundle.getString("errorFileTooBig"),
			      dsURL));
	continue;
      }
      catch (e) {
	doAlert(gStrBundle.getFormattedString("alertImportFailed", [dsURL]));
	continue;
      }

      if (fmt == validateFileFormat.CLIPPINGS_1X) {
	gAppUtils.log(String.ae_fmt("The file %S appears to be a Clippings 1.x file.", dsURL));
	legacyDataSrcURL = dsURL;
	dsURL = openLegacyClippingsFile(gCurrentFile, legacyDataSrcURL);
	if (! dsURL) {
	  legacyDataSrcURL = null;
	  continue;
	}
      }
      else if (fmt == validateFileFormat.CLIPPINGS_2) {
	gAppUtils.log(String.ae_fmt("The file %S appears to be a valid Clippings file.", dsURL));
	gCurrentFile = dlgArgs.file;
      }
      else {
	doAlert(gStrBundle.getFormattedString("alertImportFailed", [dsURL]));
	continue;
      }
    }
    else {
      // dlgArgs.fileExists is false - new file is being created
      gCurrentFile = dlgArgs.file;
    }

    try {
      initDataSrc(dsURL);
      openSuccessful = true;
    }
    catch (e) {}
  } while (! openSuccessful);

  // Opening a Clippings 1.x file for editing: import the data in the
  // Clippings 1.x file into this datasource file, leaving the original file
  // unchanged.
  if (legacyDataSrcURL) {
    doImport(true, legacyDataSrcURL);
  }

  var numItems = gClippingsSvc.numItems;
  var deck = $("entry-properties");
  deck.selectedIndex = numItems == 0 ? 1 : 2;

  gClippingsList.tree.builder.rebuild();

  if (numItems > 0) {
    gClippingsList.selectedIndex = 0;
    gClippingsList.tree.click();
    gClippingsList.tree.focus();
    gCurrentListItemIndex = 0;
  }
  
  updateItemCount();

  gClippingsListener = {
    origin: gClippingsSvc.ORIGIN_CLIPPINGS_MGR,
    newClippingCreated: function (aClippingURI) {},
    newFolderCreated: function (aFolderURI) {},
    migrationDone: function (aDataSrcURL) {},
    importDone: function (aNumItems) {}
  };
  gClippingsSvc.addListener(gClippingsListener);

  gMoveToMnu1 = $("move-to-1");
  gCopyToMnu1 = $("copy-to-1");
  gMoveToMnu2 = $("move-to-2");
  gCopyToMnu2 = $("copy-to-2");

  buildCopyToMenu(gCopyToMnu1);
  buildCopyToMenu(gMoveToMnu1);
  buildCopyToMenu(gCopyToMnu2);
  buildCopyToMenu(gMoveToMnu2);

  var clippingNameElt = $("clipping-name");
  clippingNameElt.timeout = gAppUtils.getPref("clippings.clipmgr.quickedit.update_interval");

  var url = decodeURI(dsURL);
  document.title = String.ae_fmt("%s - %s", url.substring(url.lastIndexOf("/") + 1), gStrBundle.getString("appName"));
}


function validateFileFormat(aDataSrcURL)
{
  var rv;
  var rdfSvc = Components.classes["@mozilla.org/rdf/rdf-service;1"]
                         .getService(Components.interfaces.nsIRDFService);
  var ds;
  try {
    ds = rdfSvc.GetDataSourceBlocking(aDataSrcURL);
  }
  catch (e) {
    // Any file I/O errors (file is read only, access denied, etc.) would be 
    // thrown here.
    throw e;
  }

  var ctr = Components.classes["@mozilla.org/rdf/container;1"]
                      .createInstance(Components.interfaces.nsIRDFContainer);
  var seqNode = rdfSvc.GetResource("http://clippings.mozdev.org/rdf/user-clippings-v2");
  var oldSeqNode = rdfSvc.GetResource("http://clippings.mozdev.org/rdf/user-clippings");
  try {
    ctr.Init(ds, seqNode);
    rv = validateFileFormat.CLIPPINGS_2;
  }
  catch (e) {
    try {
      ctr.Init(ds, oldSeqNode);
      rv = validateFileFormat.CLIPPINGS_1X;
    }
    catch (e) {
      rv = validateFileFormat.UNKNOWN;
    }
  }

  return rv;
}

validateFileFormat.UNKNOWN = 0;
validateFileFormat.CLIPPINGS_1X = 1;
validateFileFormat.CLIPPINGS_2 = 2;



function openLegacyClippingsFile(aFile, aLegacyDataSrcURL)
{
  var rv;  // URL of new Clippings datasource
  var url = decodeURI(aLegacyDataSrcURL);
  var filename = url.substring(url.lastIndexOf("/") + 1);
  var confirmOpen = doConfirm(gStrBundle.getFormattedString("openLegacyFmt", [filename]));
  if (! confirmOpen) {
    return rv;
  }

  var fp = Components.classes["@mozilla.org/filepicker;1"]
                     .createInstance(Components.interfaces.nsIFilePicker);
  fp.appendFilter(gStrBundle.getString("rdfExportFilterDesc"), "*.rdf");
  fp.init(window, gStrBundle.getString("saveAs"), fp.modeSave);
  fp.defaultExtension = "rdf";
  fp.defaultString = gStrBundle.getFormattedString("clippingsVersion2Filename", [filename.substring(0, filename.lastIndexOf("."))]);

  var isFileNameOK = false;
  do {
    var fpResult = fp.show();
    if (fpResult == fp.returnCancel) {
      return rv;
    }

    if (fp.fileURL.spec == aLegacyDataSrcURL) {
      gAppUtils.alertEx(gStrBundle.getString("saveAs"),
			gStrBundle.getString("errorOverwriteLegacyFile"));
    }
    else {
      isFileNameOK = true;
    }
  } while (! isFileNameOK);

  if (fpResult == fp.returnReplace) {
    fp.file.remove(false);
  }

  aFile = fp.file;
  rv = fp.fileURL.spec;

  return rv;
}


function initDataSrc(aDataSrcURL)
{
  // This function will be invoked by function init() and doSaveAs()
  var rv;
  var dsURL = aDataSrcURL;

  try {
    var ds = gClippingsSvc.getDataSource(dsURL);
  }
  catch (e if e.result == Components.results.NS_ERROR_FILE_NOT_FOUND) {
    doAlert("File not found: " + dsURL);
    throw "File not found";
  }
  catch (e if e.result == Components.results.NS_ERROR_FILE_READ_ONLY) {
    doAlert("File is read only: " + dsURL);
    throw e;
  }
  catch (e if e.result == Components.results.NS_ERROR_OUT_OF_MEMORY) {
    doAlert(gStrBundle.getString("errorOutOfMemory"));
    throw e;
  }
  catch (e if e.result == Components.results.NS_ERROR_FILE_ACCESS_DENIED) {
    doAlert(String.ae_fmt("%s: %s", gStrBundle.getString("errorAccessDenied"),
			  dsURL));
    throw e;
  }
  catch (e if e.result == Components.results.NS_ERROR_FILE_IS_LOCKED) {
    doAlert(String.ae_fmt("%s: %s", gStrBundle.getString("errorFileLocked"),
			  dsURL));
    throw e;
  }
  catch (e if e.result == Components.results.NS_ERROR_FILE_TOO_BIG) {
    doAlert(String.ae_fmt("%s: %s", gStrBundle.getString("errorFileTooBig"),
			  dsURL));
    throw e;
  }
  catch (e) {
    doAlert("Unknown error:\n\n" + e);
  }

  gClippingsSvc.processEmptyFolders();
  rv = gDataSource;
  gClippingsList.tree.database.AddDataSource(ds);
  gDataSource = ds;

  return rv;
}


function buildCopyToMenu(aMenu, aOldDataSrc)
{
  // This function is invoked during Clippings Manager initialization or
  // after migration is completed.
  if (! gDataSource) {
    gAppUtils.log("Failed to initialize Copy To or Move To menu - data source not initialized!");
    return;
  }

  if (aOldDataSrc) {
    aMenu.database.RemoveDataSource(aOldDataSrc);
  }

  aMenu.database.AddDataSource(gDataSource);
  aMenu.builder.rebuild();
}


// This is called in Clippings Manager window's unload event handler
function unload()
{
  // See HACK! note in function init() for why we're getting the width from the
  // tree's enclosing <vbox> element, and not the tree itself.
  var treeBox = $("tree-box");
  var style = window.getComputedStyle(treeBox, null);
  gAppUtils.setPref("clippings.clipmgr.treelist.width", style.width);
  
  gClippingsList.tree.builder.removeObserver(treeBuilderObserver);
  treeBuilderObserver = null;

  if (! gDataSource) {
    return;
  }

  gClippingsSvc.purgeDetachedItems();

  var retrySave;

  do {
    retrySave = false;
    try {
      gClippingsSvc.flushDataSrc();
    }
    catch (e if e.result == Components.results.NS_ERROR_NOT_INITIALIZED) {
      doAlert(gStrBundle.getString("errorSaveFailedDSNotInitialized"));
    }
    catch (e if e.result == Components.results.NS_ERROR_OUT_OF_MEMORY) {
      doAlert(gStrBundle.getString("errorOutOfMemory"));
    }
    catch (e if e.result == Components.results.NS_ERROR_FILE_ACCESS_DENIED) {
      doAlert(String.ae_fmt("%s: %s", gStrBundle.getString("errorAccessDenied"),
			    dsURL));
    }
    catch (e if e.result == Components.results.NS_ERROR_FILE_IS_LOCKED) {
      doAlert(String.ae_fmt("%s: %s", gStrBundle.getString("errorFileLocked"),
			    dsURL));
    }
    catch (e if e.result == Components.results.NS_ERROR_FILE_TOO_BIG) {
      doAlert(String.ae_fmt("%s: %s", gStrBundle.getString("errorFileTooBig"),
			    dsURL));
    }
    catch (e if e.result == Components.results.NS_ERROR_FILE_READ_ONLY) {
      doAlert(String.ae_fmt("%s: %s", gStrBundle.getString('errorFileReadOnly'),
			    dsURL));
    }
    catch (e if e.result == Components.results.NS_ERROR_FILE_DISK_FULL) {
      doAlert(String.ae_fmt("%s: %s", gStrBundle.getString("errorDiskFull"),
			    dsURL));
    }
    catch (e) {
      // Save failed for unknown reason - give the user the chance to try again
      var consoleSvc = Components.classes["@mozilla.org/consoleservice;1"].getService(Components.interfaces.nsIConsoleService);
      var msg = String.ae_fmt("Error from Clippings Manager: Error saving data source file - data source flush failed!\n\n%s", e);
      consoleSvc.logStringMessage(msg);

      retrySave = doConfirm(gStrBundle.getString("retrySave"));
    }
  }
  while (retrySave);

  gClippingsSvc.removeListener(gClippingsListener);
  gClippingsListener = null;
}


function doRecovery(aDataSrcURL, aRecoveryMode)
{
  doAlert(gStrBundle.getString('recoverFromCorruption'));

  try {
    var recoveredDataSrc = gClippingsSvc.recoverFromBackup();
    aRecoveryMode.value = doRecovery.RECOVER_FROM_BACKUP;
  }
  catch (e if e.result == Components.results.NS_ERROR_FILE_NOT_FOUND) {
    var importFromFile = doConfirm(gStrBundle.getString('msgNoBackupDoImportOption'));
    try {
      gClippingsSvc.killDataSrc();
    }
    catch (e) {
      doAlert(gStrBundle.getString('errorCannotDeleteDSFile'));
      return false;
    }

    try {
      recoveredDataSrc = gClippingsSvc.getDataSource(aDataSrcURL);
    }
    catch (e) {
      doAlert(gStrBundle.getString('errorCannotRegenerateDS'));
      return false;
    }

    if (importFromFile) {
      var result;
      do {
	result = doImport();
      } while (result == doImport.ERROR_CANNOT_IMPORT_DS
	       || result == doImport.ERROR_FILE_IO
	       || result == doImport.ERROR_FILE_UNREADABLE);

      if (result == doImport.USER_CANCEL) {
	aRecoveryMode.value = doRecovery.CREATE_BLANK_DS;
      }
      else if (result == doImport.SUCCESS) {
	aRecoveryMode.value = doRecovery.IMPORT_FROM_DS_FILE;
      }
      else {
	aRecoveryMode.value = doRecovery.FAILSAFE_CREATE_BLANK_DS;
      }
    }
    else {
      aRecoveryMode.value = doRecovery.CREATE_BLANK_DS;
    }
  }
  catch (e) {
    doAlert(gStrBundle.getString('errorRecoveryFailed'));
    return false;
  }

  return recoveredDataSrc;
}

// Constants indicating how data source recovery was performed.
doRecovery.RECOVER_FROM_BACKUP      = 1;
doRecovery.IMPORT_FROM_DS_FILE      = 2;
doRecovery.CREATE_BLANK_DS          = 3;
doRecovery.FAILSAFE_CREATE_BLANK_DS = 4;


function initReloadMenuItem() 
{
  if (! gAppUtils.isCommonDataSourceEnabled()) {
    $("reload_menuseparator").style.display = 'none';
    $("reload_menuitem").style.display = 'none';
  }
  else {
    $("migrate-menuitem").style.display = 'none';
  }
}


function initFolderMenuSeparator(aMenuPopup)
{
  var menuID = aMenuPopup.parentNode.id;
  if (gIsFolderMenuSeparatorInitialized[menuID]) {
    return;
  }

  if (gClippingsSvc.getCountSubfolders(gClippingsSvc.kRootFolderURI) > 0) {
    var clippingsRootMnuItemID = menuID + "-root";
    var clippingsRootMnuItem = $(clippingsRootMnuItemID);
    var sep = document.createElement("menuseparator");
    sep.id = menuID + "-separator";
    aMenuPopup.insertBefore(sep, clippingsRootMnuItem.nextSibling);

    gIsFolderMenuSeparatorInitialized[menuID] = true;
  }
}


function removeFolderMenuSeparator() 
{
  // This function should be called immediately before rebuilding the Move To
  // and Copy To menus, usually after creating a new folder, deleting a
  // clipping or folder, etc.
  for (var menuID in gIsFolderMenuSeparatorInitialized) {
    var popup = $(menuID + "-popup");
    var sep = $(menuID + "-separator");
    if (sep) {
      popup.removeChild(sep);
    }
    gIsFolderMenuSeparatorInitialized[menuID] = false;
  }
}


function updateItemCount()
{
  var count = gClippingsSvc.numItems;
  gStatusBar.label = gStrBundle.getFormattedString('numItems', [count]);
}


function newFolder() 
{
  updateCurrentClippingData();

  var parentFolderURI, pos;
  var selectedURI = gClippingsList.selectedURI;

  if (! selectedURI) {
    parentFolderURI = gClippingsSvc.kRootFolderURI;
  }
  else {
    parentFolderURI = gClippingsSvc.getParent(selectedURI);
    //pos = gClippingsSvc.indexOf(selectedURI);
  }

  var newFolderName = gStrBundle.getString('newFolderName');
  newFolderHelper(parentFolderURI, newFolderName, null, pos, UNDO_STACK);
}


function newFolderHelper(aParentFolderURI, aFolderName, aURI, aPos, aDestUndoStack)
{
  var newNodeURI = gClippingsSvc.createNewFolderEx(aParentFolderURI, aURI, aFolderName, aPos, false, gClippingsSvc.ORIGIN_CLIPPINGS_MGR);

  gClippingsList.selectedURI = newNodeURI;
  gClippingsList.ensureURIIsVisible(newNodeURI);
  gClippingsList.tree.click();

  var deck = $("entry-properties");
  if (deck.selectedIndex != 0) {
    deck.selectedIndex = 0;
  }

  var folderName = $("clipping-name");
  folderName.select();
  folderName.focus();

  // Undo / redo
  var state = {
    action:  ACTION_CREATENEWFOLDER, 
    uri:     newNodeURI, 
    name:    aFolderName,
    pos:     aPos,
    parentFolderURI: aParentFolderURI
  };

  if (aDestUndoStack == UNDO_STACK) {
    gUndoStack.push(state);
    gAppUtils.log(String.ae_fmt("New entry %S added to undo stack", aFolderName));
  }
  else if (aDestUndoStack == REDO_STACK) {
    gRedoStack.push(state);
    gAppUtils.log(String.ae_fmt("New entry %S added to redo stack", aFolderName));
  }

  gIsClippingsDirty = true;

  removeFolderMenuSeparator();
  gMoveToMnu1.builder.rebuild();
  gCopyToMnu1.builder.rebuild();
  gMoveToMnu2.builder.rebuild();
  gCopyToMnu2.builder.rebuild();
  
  commit();
}


function newClipping() 
{
  updateCurrentClippingData();

  var parentFolderURI, pos;
  var selectedURI = gClippingsList.selectedURI;

  if (! selectedURI) {
    parentFolderURI = gClippingsSvc.kRootFolderURI;
  }
  else {
    parentFolderURI = gClippingsSvc.getParent(selectedURI);
    //pos = gClippingsSvc.indexOf(selectedURI);
  }

  newClippingHelper(parentFolderURI,
		    gStrBundle.getString('newClippingName'), "", null, pos,
		    UNDO_STACK);
}


function newClippingHelper(aParentFolder, aName, aText, aURI, aPos, aDestUndoStack)
{
  var newNodeURI = gClippingsSvc.createNewClippingEx(aParentFolder, aURI, aName, aText, aPos, true);

  if (! newNodeURI) {
    doAlert(gStrBundle.getString('errorCannotCreate'));
    return;
  }
  
  gClippingsList.selectedURI = newNodeURI;
  gClippingsList.ensureURIIsVisible(newNodeURI);
  gClippingsList.tree.click();

  var deck = $("entry-properties");
  if (deck.selectedIndex != 0) {
    deck.selectedIndex = 0;
  }

  var clippingName = $("clipping-name");
  clippingName.select();
  clippingName.focus();

  var state = {action: ACTION_CREATENEW, uri: newNodeURI, name: aName, text: aText, parentFolderURI: aParentFolder, pos: aPos};
  if (aDestUndoStack == UNDO_STACK) {
    gUndoStack.push(state);
    gAppUtils.log(String.ae_fmt("New entry %S added to undo stack", aName));
  }
  else if (aDestUndoStack == REDO_STACK) {
    gRedoStack.push(state);
    gAppUtils.log(String.ae_fmt("New entry %S added to redo stack", aName));
  }

  gIsClippingsDirty = true;
  commit();
}


function pasteClippingAsNew()
{
  updateCurrentClippingData();

  var txt = gAppUtils.getTextFromClipboard();
  if (! txt) {
    gAppUtils.beep();
    gStatusBar.label = gStrBundle.getString('errorNoClipboardContent');
    return;
  }

  var parentFolderURI;
  var selectedURI = gClippingsList.selectedURI;
  if (! selectedURI) {
    parentFolderURI = gClippingsSvc.kRootFolderURI;
  }
  else {
    parentFolderURI = gClippingsSvc.getParent(selectedURI);
  }

  var newNodeURI = gClippingsSvc.newClippingFromText(txt, false, window,
						     parentFolderURI, true);
  if (! newNodeURI) {
    doAlert(gStrBundle.getString('errorCannotCreate'));
    return;
  }

  gClippingsList.selectedURI = newNodeURI;
  gClippingsList.ensureURIIsVisible(newNodeURI);
  gClippingsList.tree.click();

  var deck = $("entry-properties");
  if (deck.selectedIndex != 0) {
    deck.selectedIndex = 0;
  }

  var clippingName = $("clipping-name");
  clippingName.select();
  clippingName.focus();

  gUndoStack.push({action: ACTION_CREATENEW, uri: newNodeURI,
	           parentFolderURI: parentFolderURI,
		   name: gClippingsSvc.getName(newNodeURI),
		   text: gClippingsSvc.getText(newNodeURI)});
  gAppUtils.log(String.ae_fmt("New entry added to undo stack\nName = %S", 
			       gClippingsSvc.getName(newNodeURI)));

  gIsClippingsDirty = true;
  commit();
}


function copyToFolder(aDestFolderURI)
{
  // We have to do this, otherwise if the Move button was clicked, it will
  // have the sunken look again when the mouse is hovered over it later.
  window.setTimeout("copyToFolderEx('" + aDestFolderURI + "');", 10);
}


function copyToFolderEx(aDestFolderURI)
{
  updateCurrentClippingData();

  var destFolderURI = aDestFolderURI;
  var itemURI = gClippingsList.selectedURI;
  var parentFolderURI = gClippingsSvc.getParent(itemURI);

  // Prevent infinite recursion due to copying a folder into its own subfolder.
  if (gClippingsSvc.isFolder(itemURI)) {
    var parentURI = gClippingsSvc.getParent(aDestFolderURI);
    while (parentURI && parentURI != gClippingsSvc.kRootFolderURI) {
      if (parentURI == itemURI) {
	gAppUtils.beep();
	gAppUtils.log("Cannot copy a folder into a subfolder of itself!");
	return;
      }
      parentURI = gClippingsSvc.getParent(parentURI);
    }
  }

  copyToFolderHelper(itemURI, parentFolderURI, destFolderURI, null, null, UNDO_STACK, false);
}


function copyToFolderHelper(aItemURI, aSrcFolderURI, aDestFolderURI, aDestItemURI, aDestPos, aDestUndoStack, aSelectCopiedItem)
{
  var prevIndex = gClippingsList.selectedIndex;
  var newURI = gClippingsSvc.copyTo(aItemURI, aDestFolderURI, aDestItemURI,
				    aDestPos, false,
				    gClippingsSvc.ORIGIN_CLIPPINGS_MGR);

  gAppUtils.log(String.ae_fmt("Copy completed.  Name of copied item: %S\nParent folder of copied item: %S", gClippingsSvc.getName(aItemURI), gClippingsSvc.getName(aDestFolderURI)));

  removeFolderMenuSeparator();
  gMoveToMnu1.builder.rebuild();
  gCopyToMnu1.builder.rebuild();
  gMoveToMnu2.builder.rebuild();
  gCopyToMnu2.builder.rebuild();

  // The Clippings tree context menu doesn't close automatically after the
  // copy or move operation.
  $("clippings-list-context").hidePopup();
  $("toolbar-move-popup").hidePopup();

  var state = {
    action:       ACTION_COPYTOFOLDER,
    copyURI:      newURI,
    originalURI:  aItemURI,
    srcFolder:    aSrcFolderURI
  };

  if (aDestUndoStack == UNDO_STACK) {
    gUndoStack.push(state);
  }
  else if (aDestUndoStack == REDO_STACK) {
    gRedoStack.push(state);
  }

  if (aSelectCopiedItem) {
    gClippingsList.selectedURI = newURI;
    gClippingsList.ensureURIIsVisible(newURI);
  }
  else {
    var numRows = gClippingsList.getRowCount();
    if (prevIndex == numRows) {  // Copied item was last list item.
      gClippingsList.selectedIndex = numRows - 1;
      gClippingsList.ensureIndexIsVisible(numRows - 1);
    }
    else {
      gClippingsList.selectedIndex = prevIndex;
      gClippingsList.ensureIndexIsVisible(prevIndex);
    }
  }

  updateDisplay();
  updateItemCount();
  gIsClippingsDirty = true;
  commit();
}


function moveToFolder(aDestFolderURI)
{
  // We have to do this, otherwise if the Move button was clicked, it will
  // have the pressed-down look again when the mouse is hovered over it later.
  window.setTimeout("moveToFolderEx('" + aDestFolderURI + "');", 10);
}


function moveToFolderEx(aDestFolderURI)
{
  updateCurrentClippingData();

  var destFolderURI = aDestFolderURI;
  var itemURI = gClippingsList.selectedURI;

  if (gClippingsSvc.isFolder(itemURI) && itemURI == destFolderURI) {
    gAppUtils.beep();
    gAppUtils.log("Cannot move a folder into itself!");
    return;
  }

  // It is pointless to move an item into the same folder.
  var parentFolderURI = gClippingsSvc.getParent(itemURI);
  if (parentFolderURI == destFolderURI) {
    gAppUtils.beep();
    gAppUtils.log("The source and destination folders are the same!");
    return;
  }

  // Prevent infinite recursion due to moving a folder into its own subfolder.
  if (gClippingsSvc.isFolder(itemURI)) {
    var parentURI = gClippingsSvc.getParent(aDestFolderURI);
    while (parentURI && parentURI != gClippingsSvc.kRootFolderURI) {
      if (parentURI == itemURI) {
	gAppUtils.beep();
	gAppUtils.log("Cannot move a folder into a subfolder of itself!");
	return;
      }
      parentURI = gClippingsSvc.getParent(parentURI);
    }
  }

  moveToFolderHelper(itemURI, parentFolderURI, destFolderURI, null, null, UNDO_STACK, false);
}


function moveToFolderHelper(aItemURI, aSrcFolderURI, aDestFolderURI, aDestItemURI, aDestPos, aDestUndoStack, aSelectMovedItem)
{
  var pos = gClippingsSvc.indexOf(aItemURI);
  var prevIndex = gClippingsList.selectedIndex;
  var newURI = gClippingsSvc.copyTo(aItemURI, aDestFolderURI, aDestItemURI,
				    aDestPos, true,
				    gClippingsSvc.ORIGIN_CLIPPINGS_MGR);

  gAppUtils.log(String.ae_fmt("Move completed.  Name of moved item: %S\nParent folder the item was moved to: %S", gClippingsSvc.getName(newURI), gClippingsSvc.getName(aDestFolderURI)));

  removeFolderMenuSeparator();
  gMoveToMnu1.builder.rebuild();
  gCopyToMnu1.builder.rebuild();
  gMoveToMnu2.builder.rebuild();
  gCopyToMnu2.builder.rebuild();

  // The Clippings tree context menu doesn't close automatically after the
  // copy or move operation.
  $("clippings-list-context").hidePopup();
  $("toolbar-move-popup").hidePopup();

  var state = {
    action: ACTION_MOVETOFOLDER,
    uri:    newURI,
    originalFolder: aSrcFolderURI,
    originalURI: aItemURI,
    originalPos: pos
  };

  if (aDestUndoStack == UNDO_STACK) {
    gUndoStack.push(state);
  }
  else if (aDestUndoStack == REDO_STACK) {
    gRedoStack.push(state);
  }

  if (aSelectMovedItem) {
    // The following will not work if the parent of the item to select is
    // not expanded.
    gClippingsList.selectedURI = newURI;
    gClippingsList.ensureURIIsVisible(newURI);
  }
  else {
    var numRows = gClippingsList.getRowCount();
    if (prevIndex == numRows) {  // Moved item was last list item.
      gClippingsList.selectedIndex = numRows - 1;
      gClippingsList.ensureIndexIsVisible(numRows - 1);
    }
    else {
      gClippingsList.selectedIndex = prevIndex;
      gClippingsList.ensureIndexIsVisible(prevIndex);
    }
  }

  updateDisplay();
  gIsClippingsDirty = true;
  commit();
}



function updateEditStatus() 
{
  gStatusBar.label = gStrBundle.getString("editEntry");
}


function commit()
{
  return;
}


function doSaveAs()
{
  // Saves the current file with a new name, and continue editing with this
  // new file.
  updateCurrentClippingData();

  var file = gCurrentFile.QueryInterface(Components.interfaces.nsIFile);
  var parentDir = file.parent;
  var fph = Components.classes["@mozilla.org/network/protocol;1?name=file"]
                      .createInstance(Components.interfaces
				                .nsIFileProtocolHandler);
  var currentFileURL = fph.getURLSpecFromFile(file);
  var fp = Components.classes["@mozilla.org/filepicker;1"]
                     .createInstance(Components.interfaces.nsIFilePicker);
  fp.init(window, gStrBundle.getString("saveAs"), fp.modeSave);
  fp.defaultExtension = "rdf";
  fp.defaultString = file.leafName;
  fp.appendFilter(gStrBundle.getString("rdfImportFilterDesc"), "*.rdf");

  var fpResult = fp.show();
  if (fpResult == fp.returnCancel) {
    return;
  }
  
  saveClippings(true, true, false);

  var newFile = fp.file.QueryInterface(Components.interfaces.nsIFile);
  var newFileURL = fp.fileURL.spec;
  var newFilePath = newFile.path;
  var newFileName = newFile.leafName;

  if (fpResult == fp.returnReplace) {
    if (newFileURL == currentFileURL) {
      gAppUtils.log("doSaveAs(): Save As with same file name and in same folder");
      return;
    }

    gAppUtils.log("doSaveAs(): Deleting existing file: " + newFilePath);
    fp.file.remove(false);  // Remove existing file
  }
 
  file.copyTo(parentDir, newFileName);
  gCurrentFile = newFile;
  gClippingsSvc.reset();
  gClippingsList.tree.database.RemoveDataSource(gDataSource);

  try {
    var oldDS = initDataSrc(newFileURL);
  }
  catch (e) {}

  buildCopyToMenu(gCopyToMnu1, oldDS);
  buildCopyToMenu(gMoveToMnu1, oldDS);
  buildCopyToMenu(gCopyToMnu2, oldDS);
  buildCopyToMenu(gMoveToMnu2, oldDS);

  document.title = String.ae_fmt("%s - %s", newFileName, gStrBundle.getString("appName"));
}


function saveClippings(aSuppressStatusMsgs, aForceSave, aDoBackup)
{
  updateCurrentClippingData();

  if (gSaveInProgress || (!gIsClippingsDirty && !aForceSave)) {
    if (gSaveInProgress) {
      gAppUtils.beep();
      gAppUtils.log("saveClippings() aborted because another commit/save is in progress");
    }
    return;
  }

  if (! aSuppressStatusMsgs) {
    gStatusBar.label = gStrBundle.getString("saveProgress");
  }

  var msg = gStrBundle.getString("saveCompleted");
  try {
    gSaveInProgress = true;
    gClippingsSvc.flushDataSrcEx(aDoBackup);
    gIsClippingsDirty = false;

    if (aSuppressStatusMsgs) {
      return;
    }
  }
  catch (e) {
    msg = gStrBundle.getString("errorSaveFailed");
  }
  finally {
    gSaveInProgress = false;
  }

  gStatusBar.label = msg;
}


function reload()
{
  // WARNING:
  // Do not open any windows or dialog boxes from within this function;
  // otherwise Clipping Manager's reload event handler will be invoked again
  // when the window or dialog box is closed!

  if (gJustMigrated) {
    // Not necessary to reload if Migration Wizard has just completed.
    // Also, clear the `dirty' flag because it was ignored when datasource was
    // not migrated from user profile directory, and leaving it set would cause
    // the error message below to appear unnecessarily.
    gJustMigrated = false;
    gIsClippingsDirty = false;
    return;
  }

  if (gSaveInProgress || !gAppUtils.isCommonDataSourceEnabled()) {
    updateDisplay();
    return;
  }

  if (gIsClippingsDirty) {
    gAppUtils.beep();
    gAppUtils.log("function reload(): Previous commit failed; changes may be overwritten by changes from other host app\ngIsClippingsDirty = " + gIsClippingsDirty + "\nResetting gIsClippingsDirty flag");
    gIsClippingsDirty = false;
  }

  // currIndex == -1 if nothing was selected
  var currIndex = gClippingsList.selectedIndex;
  currIndex = currIndex == -1 ? 0 : currIndex;

  try {
    gClippingsSvc.refreshDataSrc();
  }
  catch (e) {
    gAppUtils.beep();
    gAppUtils.log("function reload(): Reload failed!\n\n" + e);
    gStatusBar.label = "Reload Failed!";
    return;
  }

  gClippingsList.tree.builder.rebuild();
  updateItemCount();

  // Selection on tree list disappears after rebuild.  Restore it.
  var numRows = gClippingsList.getRowCount();
  if (numRows == 0) {
    $("entry-properties").selectedIndex = 1;
    gCurrentListItemIndex = -1;
  }
  else if (currIndex == numRows) {  // Deleted item was last list item
    gClippingsList.selectedIndex = numRows - 1;
    gClippingsList.ensureIndexIsVisible(numRows - 1);
    updateDisplay();
  }
  else {
    gClippingsList.selectedIndex = currIndex;
    gClippingsList.ensureIndexIsVisible(currIndex);
    updateDisplay();
  }
}


function deleteClipping(aDelKeyPressed) 
{
  if (aDelKeyPressed) {
    var delKeyEnabled = gAppUtils.getPref("clippings.clipmgr.enable_delete_key", true);
    if (! delKeyEnabled) {
      return;
    }
  }

  if (gClippingsList.getRowCount() == 0) {
    gAppUtils.beep();
    gStatusBar.label = gStrBundle.getString("msgNothingToDelete");
    return;
  }

  updateCurrentClippingData();

  var uri = gClippingsList.selectedURI;
  if (! uri) {
    return;
  }

  if (gClippingsSvc.isEmptyClipping(uri)) {
    gAppUtils.beep();
    return;
  }

  deleteClippingHelper(uri, UNDO_STACK);
}


function deleteClippingHelper(aURI, aDestUndoStack)
  // NOTE - This deletes both clippings and folders
{
  var deletedItemIndex = gClippingsList.selectedIndex;
  var state;
  var pos = gClippingsSvc.indexOf(aURI);

  if (gClippingsSvc.isClipping(aURI)) {
    state = {
      action: ACTION_DELETECLIPPING,
      uri:    aURI, 
      name:   gClippingsSvc.getName(aURI),
      text:   gClippingsSvc.getText(aURI),
      key:    gClippingsSvc.getShortcutKey(aURI),
      pos:    pos,
      parentFolderURI: gClippingsSvc.getParent(aURI)
    };

    try {
      gClippingsSvc.remove(aURI);
    }
    catch (e) {
      doAlert(gStrBundle.getString("errorCannotDelete"));
      return;
    }
  }
  else if (gClippingsSvc.isFolder(aURI)) {
    if (gClippingsSvc.getCount(aURI) > 0) {
      gAppUtils.log("Removing non-empty folder");
      state = {
	action: ACTION_DELETEFOLDER,
	name:   gClippingsSvc.getName(aURI),
	parentFolderURI: gClippingsSvc.getParent(aURI)
      };

      try {
	gClippingsSvc.detachFromFolder(aURI);
      }
      catch (e) {
	doAlert(gStrBundle.getString("errorCannotDelete"));
	return;
      }
    }
    else {
      gAppUtils.log("Removing empty folder");
      state = {
	action: ACTION_DELETEEMPTYFOLDER,
	name:   gClippingsSvc.getName(aURI),
	parentFolderURI: gClippingsSvc.getParent(aURI)
      };

      try {
	gClippingsSvc.remove(aURI);
      }
      catch (e) {
	doAlert(gStrBundle.getString("errorCannotDelete"));
	return;
      }
    }

    state.uri = aURI;
    state.pos = pos;
  }

  if (aDestUndoStack == UNDO_STACK) {
    gUndoStack.push(state);
    gAppUtils.log(String.ae_fmt("Deleted entry %S (URI %s added to undo stack)\nPosition: %d", state.name, state.uri, state.pos));
  }
  else if (aDestUndoStack == REDO_STACK) {
    gRedoStack.push(state);
    gAppUtils.log(String.ae_fmt("Deleted entry %S (URI %s added to redo stack)", state.name, uri));
  }

  gIsClippingsDirty = true;
  removeFolderMenuSeparator();
  gClippingsList.tree.builder.rebuild();

  updateItemCount();

  var numRows = gClippingsList.getRowCount();
  if (numRows == 0) {
    $("entry-properties").selectedIndex = 1;
    gCurrentListItemIndex = -1;
  }
  else if (deletedItemIndex == numRows) {  // Deleted item was last list item
    gClippingsList.selectedIndex = numRows - 1;
    gClippingsList.ensureIndexIsVisible(numRows - 1);
    updateDisplay();
  }
  else {
    gClippingsList.selectedIndex = deletedItemIndex;
    gClippingsList.ensureIndexIsVisible(deletedItemIndex);
    updateDisplay();
  }

  commit();
}


function updateCurrentEntryStatus()
{
  if (gClippingsList.getRowCount() == 0) {
    return;
  }

  updateItemCount();

  var deck = $("entry-properties");
  if (deck.selectedIndex != 0) {
    deck.selectedIndex = 0;
  }
}


function updateDisplay(aSuppressUpdateSelection)
{
  if (gClippingsList.getRowCount() == 0) {
    return;
  }

  var clippingName = $("clipping-name");
  var clippingNameLabel = $("clipping-name-label");
  var clippingText = $("clipping-text");
  var clippingTextLabel = $("clipping-text-label");
  var clippingKey = $("clipping-key");
  var clippingKeyLabel = $("clipping-key-label");
  var shortcutKeyMiniHelp = $("shortcut-key-minihelp");

  var uri = gClippingsList.selectedURI;
  if (! uri) {
    // Nothing selected - so just select whatever is at the current index.
    var numRows = gClippingsList.getRowCount();
    if (gCurrentListItemIndex > (numRows - 1)) {
      gCurrentListItemIndex = numRows - 1;
    }

    gClippingsList.selectedIndex = gCurrentListItemIndex;
    uri = gClippingsList.selectedURI;
  }

  if (gClippingsSvc.isFolder(uri)) {
    clippingNameLabel.disabled = false;
    clippingName.disabled = false;
    clippingTextLabel.disabled = true;
    clippingText.disabled = true;
    shortcutKeyMiniHelp.disabled = true;
      
    clippingKey.selectedIndex = 0;
    clippingKeyLabel.disabled = true;
    clippingKey.disabled = true;
  }
  // Special handling of the dummy item in an empty folder
  else if (gClippingsSvc.isEmptyClipping(uri)) {
    clippingNameLabel.disabled = true;
    clippingName.disabled = true;
    clippingTextLabel.disabled = true;
    clippingText.disabled = true;
    shortcutKeyMiniHelp.disabled = true;

    clippingKey.selectedIndex = 0;
    clippingKeyLabel.disabled = true;
    clippingKey.disabled = true;    
  }
  else {
    clippingNameLabel.disabled = false;
    clippingName.disabled = false;
    clippingTextLabel.disabled = false;
    clippingText.disabled = false;
    clippingKeyLabel.disabled = false;
    clippingKey.disabled = false;
    shortcutKeyMiniHelp.disabled = false;
  }

  clippingName.value = gClippingsSvc.getName(uri);
  clippingText.value = gClippingsSvc.getText(uri);

  var key;
  if (gClippingsSvc.isClipping(uri) && (key = gClippingsSvc.getShortcutKey(uri))) {
    for (var i = 0; i < clippingKey.menupopup.childNodes.length; i++) {
      if (clippingKey.menupopup.childNodes[i].label == key) {
	clippingKey.selectedIndex = i;
	break;
      }
    }
  }
  else {
    clippingKey.selectedIndex = 0;
  }

  gCurrentListItemIndex = gClippingsList.selectedIndex;

  if (! aSuppressUpdateSelection) {
    gClippingsList.ensureIndexIsVisible(gClippingsList.selectedIndex);
  }
}


function updateName(aName)
{
  if (gCurrentListItemIndex == -1) {
    return;
  }

  var uri = gClippingsList.getURIAtIndex(gCurrentListItemIndex);
  updateNameHelper(uri, aName, UNDO_STACK);
}


function updateNameHelper(aURI, aName, aDestUndoStack)
{
  var oldName = gClippingsSvc.getName(aURI);
  if (oldName == aName) {
    return;
  }

  var state = {action: ACTION_EDITNAME, uri: aURI, name: oldName, text: null};
  if (aDestUndoStack == UNDO_STACK) {
    gUndoStack.push(state);
    gAppUtils.log(String.ae_fmt("In function updateNameHelper(): Old Entry name: %S\nOld entry name added to undo stack", oldName));
  }
  else if (aDestUndoStack == REDO_STACK) {
    gRedoStack.push(state);
    gAppUtils.log(String.ae_fmt("In function updateNameHelper(): Old Entry name: %S\nOld entry name added to redo stack", oldName));
  }

  gClippingsSvc.setName(aURI, aName);
  gIsClippingsDirty = true;
  commit();
}


function updateText(aText)
{
  if (gCurrentListItemIndex == -1) {
    return;
  }
  var uri = gClippingsList.getURIAtIndex(gCurrentListItemIndex);
  if (gClippingsSvc.isFolder(uri)) {
    // Folders don't have a `text' predicate, so just ignore
    return;
  }

  updateTextHelper(uri, aText, UNDO_STACK);
}


function updateTextHelper(aURI, aText, aDestUndoStack)
{
  var oldText = gClippingsSvc.getText(aURI);
  if (oldText == aText) {
    return;
  }

  // DEBUGGING
  var name = gClippingsSvc.getName(aURI);
  // END DEBUGGING

  var state = {action: ACTION_EDITTEXT, uri: aURI, name: name, text: oldText};
  if (aDestUndoStack == UNDO_STACK) {
    gUndoStack.push({action:ACTION_EDITTEXT, uri:aURI, name:name, text:oldText});
    // We shouldn't care what the old name is.
    // Non-debug invocation to push() should be as follows:
    /***
	gUndoStack.push(ACTION_EDITTEXT, aURI, null, oldText);
    ***/
    gAppUtils.log(String.ae_fmt("In function updateTextHelper(): Entry name: %S\nOld text before edit added to undo stack\noldText = %S", name, oldText));
  }
  else if (aDestUndoStack == REDO_STACK) {
    gRedoStack.push(state);
    gAppUtils.log(String.ae_fmt("In function updateTextHelper(): Added to redo stack: oldText == %S", oldText));
  }

  gClippingsSvc.setText(aURI, aText);
  gIsClippingsDirty = true;
  commit();
}


function updateCurrentClippingData()
{
  if (gClippingsSvc.numItems > 0) {
    updateName($("clipping-name").value);

    var uri = gClippingsList.getURIAtIndex(gCurrentListItemIndex);
    if (gClippingsSvc.isClipping(uri)) {
      updateText($("clipping-text").value);
    }
  }
}


function moveUp()
{
  var currentURI = gClippingsList.selectedURI;
  var p = gClippingsSvc.getParent(currentURI);
  var i = gClippingsSvc.indexOf(currentURI);
  var max = gClippingsSvc.getCount(p);

  if (1 < i && i <= max) {
    moveEntry(currentURI, p, i, i - 1, UNDO_STACK);
    gClippingsList.selectedURI = currentURI;
    updateDisplay();
  }
  else if (i == 1) {
    gAppUtils.beep();
    gStatusBar.label = "Already at first item of folder";
  }
}


function moveDown()
{
  var currentURI = gClippingsList.selectedURI;
  var p = gClippingsSvc.getParent(currentURI);
  var i = gClippingsSvc.indexOf(currentURI);
  var max = gClippingsSvc.getCount(p);

  if (1 <= i && i < max) {
    moveEntry(currentURI, p, i, i + 1, UNDO_STACK);
    gClippingsList.selectedURI = currentURI;
    updateDisplay();
  }
  else if (i == max) {
    gAppUtils.beep();
    gStatusBar.label = "Already at last item of folder";
  }
}


function moveEntry(aURI, aParentFolderURI, aOldPos, aNewPos, aDestUndoStack)
  // aOldPos and aNewPos are 1-based (NOT zero-based) indices
{
  var state = {action: ACTION_CHANGEPOSITION, uri: aURI,
	       currPos: aNewPos, prevPos: aOldPos,
	       parentFolderURI: aParentFolderURI};

  if (aDestUndoStack == UNDO_STACK) {
    gUndoStack.push(state);
  }
  else if (aDestUndoStack == REDO_STACK) {
    gRedoStack.push(state);
  }

  gAppUtils.log(String.ae_fmt("In function moveEntry(): aOldPos=%d; aNewPos=%d", aOldPos, aNewPos));

  gClippingsSvc.changePosition(aParentFolderURI, aOldPos, aNewPos);
  gIsClippingsDirty = true;
  removeFolderMenuSeparator();  // Rebuild folder menu separator
  commit();
}


function doExport()
{
  window.openDialog("chrome://clippings/content/export.xul", "export_dlg", "dialog,modal,centerscreen", gClippingsSvc);
}


function doImport(aNoFilePrompt, aDataSrcURL)
{
  var url, path;

  if (aNoFilePrompt) {
    url = aDataSrcURL;
    var io = Components.classes["@mozilla.org/network/io-service;1"]
                       .getService(Components.interfaces.nsIIOService);
    var fh = io.getProtocolHandler("file")
               .QueryInterface(Components.interfaces.nsIFileProtocolHandler);
    path = fh.getFileFromURLSpec(url);
  }
  else {
    var fp = Components.classes["@mozilla.org/filepicker;1"]
                       .createInstance(Components.interfaces.nsIFilePicker);
    fp.init(window, gStrBundle.getString("dlgTitleImportClippings"), fp.modeOpen);
    fp.appendFilter(gStrBundle.getString("rdfImportFilterDesc"), "*.rdf");

    var dlgResult = fp.show();
    if (dlgResult != fp.returnOK) {
      return doImport.USER_CANCEL;
    }

    url = fp.fileURL.QueryInterface(Components.interfaces.nsIURI).spec;
    path = fp.file.QueryInterface(Components.interfaces.nsIFile).path;
  }

  gStatusBar.label = gStrBundle.getString("importBegin");

  try {
    var importDSRootCtr = {};
    var numImported = gClippingsSvc.importFromFile(url, false, false, importDSRootCtr);
  }
  catch (e if e.result == Components.results.NS_ERROR_NOT_INITIALIZED) {
    doAlert(gStrBundle.getString('alertImportFailedNoDS'));
    gStatusBar.label = "";
    return doImport.ERROR_UNINITIALIZED_DS;
  }
  catch (e if e.result == Components.results.NS_ERROR_OUT_OF_MEMORY) {
    doAlert(gStrBundle.getString("errorOutOfMemory"));
    gStatusBar.label = "";
    return doImport.ERROR_UNEXPECTED;
  }
  catch (e if e.result == Components.results.NS_ERROR_FILE_ACCESS_DENIED) {
    doAlert(String.ae_fmt("%s: %S", gStrBundle.getString("errorAccessDenied"),
			  path));
    gStatusBar.label = "";
    return doImport.ERROR_FILE_IO;
  }
  catch (e if e.result == Components.results.NS_ERROR_FILE_IS_LOCKED) {
    doAlert(String.ae_fmt("%s: %S", gStrBundle.getString("errorFileLocked"),
			  path));
    gStatusBar.label = "";
    return doImport.ERROR_FILE_IO;
  }
  catch (e) {
    gAppUtils.log(e);

    var err = gStrBundle.getFormattedString("alertImportFailed", [path]);
    doAlert(err);
    gStatusBar.label = "";
    return doImport.ERROR_FILE_UNREADABLE;
  }

  importDSRootCtr = importDSRootCtr.value;

  // Handle empty RDF files
  if (numImported == 0) {
    gStatusBar.label = gStrBundle.getString("msgNoItems");
    return doImport.SUCCESS;
  }

  // Handle conflicting shortcut keys
  var conflictingKeys = false;
  var importFlag = gClippingsSvc.IMPORT_REPLACE_CURRENT;

  try {
    conflictingKeys = gClippingsSvc.hasConflictingShortcutKeys(importDSRootCtr);
  }
  catch (e) {}

  if (conflictingKeys) {
    var prtSvc = Components.classes["@mozilla.org/embedcomp/prompt-service;1"]
                           .getService(Components.interfaces.nsIPromptService);
    var btnFlags = prtSvc.STD_YES_NO_BUTTONS + prtSvc.BUTTON_POS_1_DEFAULT;
    var choice = prtSvc.confirmEx(window, gStrBundle.getString("appName"), gStrBundle.getString("shortcutKeyConflict"), btnFlags, "", "", "", "", {});

    if (choice == 1) {
      importFlag = gClippingsSvc.IMPORT_KEEP_CURRENT;
    }
  }

  try {
    gClippingsSvc.importShortcutKeys(importDSRootCtr, importFlag);
  }
  catch (e) {}

  // Append the "empty" clipping to any empty folders that were imported
  gClippingsSvc.processEmptyFolders();

  var deck = $("entry-properties");
  if (deck.selectedIndex != 0) {
    if (gClippingsList.getRowCount() > 0) {
      deck.selectedIndex = 0;
      gClippingsList.selectedIndex = 0;
      gClippingsList.tree.click();
    }
  }    
  // Status bar msg is overwritten in listbox.click() call, so redisplay
  // status of import.
  gStatusBar.label = String.ae_fmt("%s%s",
				   gStrBundle.getString("importBegin"),
				   gStrBundle.getString("importDone"));
  try {
    gClippingsSvc.flushDataSrc();
  }
  catch (e) {
    // Don't do anything for now - try again when closing Clippings Manager.
  }

  return doImport.SUCCESS;
}

// Return values of doImport()
doImport.USER_CANCEL             = 0;
doImport.SUCCESS                 = 1;
doImport.ERROR_FILE_IO           = 2;
doImport.ERROR_UNINITIALIZED_DS  = 3;
doImport.ERROR_FILE_UNREADABLE   = 4;
doImport.ERROR_CANNOT_IMPORT_DS  = 5;
doImport.ERROR_UNEXPECTED        = 15;



function initClippingsListPopup()
{
  var numListItems = gClippingsList.getRowCount();
  if (numListItems == 0) {
    return false;  // Don't show popup menu if list box is empty
  }

  var clippingsListCxt = $("clippings-list-context");
  var uri = gClippingsList.selectedURI;
  var isEmptyClipping = gClippingsSvc.isEmptyClipping(uri);

  for (var i = 0; i < clippingsListCxt.childNodes.length; i++) {
    clippingsListCxt.childNodes[i].setAttribute("disabled", isEmptyClipping);
  }

  return true;
}


function initMoveMenuPopup()
{
  var moveToMnu2 = $("move-to-2");
  var copyToMnu2 = $("copy-to-2");
  var uri = gClippingsList.selectedURI;
  var boolFlag  = gClippingsSvc.numItems == 0 || gClippingsSvc.isEmptyClipping(uri);

  moveToMnu2.setAttribute("disabled", boolFlag);
  copyToMnu2.setAttribute("disabled", boolFlag);

  return true;
}


function undo()
{
  updateCurrentClippingData();

  if (gUndoStack.length == 0) {
    gAppUtils.beep();
    gStatusBar.label = gStrBundle.getString("msgNothingToUndo");
    return;
  }

  var undo = gUndoStack.pop();
  var srcFolder;  // Used for undoing move or copy
  var pos;        // Used for undoing folder or clipping creation

  if (! gClippingsSvc.exists(undo.uri) && undo.action != ACTION_DELETECLIPPING
      && undo.action != ACTION_DELETEFOLDER && undo.action != ACTION_COPYTOFOLDER) {
    gAppUtils.beep();
    gStatusBar.label = gStrBundle.getString("cantUndo");
    return;
  }

  if (undo.action == ACTION_DELETECLIPPING) {
    gClippingsSvc.createNewClippingEx(undo.parentFolderURI, undo.uri,
				      undo.name, undo.text, undo.pos, true);

    if (undo.key) {
      try {
	gClippingsSvc.setShortcutKey(undo.uri, undo.key);
      }
      catch (e) {
	// Shortcut key already assigned to a clipping - theoretically, this is
	// an unreachable case because the shortcut key assignment would have
	// been undone before undoing deletion.
	gAppUtils.beep();
	gAppUtils.log(String.ae_fmt("Cannot restore shortcut key: the key %S is assigned to another clipping", undo.key));
      }
    }

    gIsClippingsDirty = true;
    commit();

    gClippingsList.selectedURI = undo.uri;
    gClippingsList.ensureURIIsVisible(undo.uri);
    updateDisplay();

    gAppUtils.log(String.ae_fmt("Deletion undo\nEntry name: %S; entry URI: %S", undo.name, undo.uri));

    var state = {
      action: ACTION_DELETECLIPPING, 
      uri:    undo.uri,
      name:   undo.name, 
      text:   undo.text,
      key:    undo.key || "",
      pos:    undo.pos
    };

    gRedoStack.push(state);
  }
  else if (undo.action == ACTION_EDITTEXT) {
    updateTextHelper(undo.uri, undo.text, REDO_STACK);
    gClippingsList.selectedURI = undo.uri;
    updateDisplay();

    var clippingText = $("clipping-text");
    clippingText.focus();
    clippingText.select();

    gAppUtils.log(String.ae_fmt("Entry text edit undo\nEntry name: %S; entry URI: %S; oldText: %S", undo.name, undo.uri, undo.text));
  }
  else if (undo.action == ACTION_EDITNAME) {
    updateNameHelper(undo.uri, undo.name, REDO_STACK);
    gClippingsList.selectedURI = undo.uri;
    updateDisplay();

    var clippingName = $("clipping-name");
    clippingName.focus();
    clippingName.select();

    gAppUtils.log(String.ae_fmt("Entry name edit undo\nEntry name: %S; entry URI: %S", undo.name, undo.uri));
  }
  else if (undo.action == ACTION_CREATENEW) {
    var key = gClippingsSvc.getShortcutKey(undo.uri);
    if (key) { gAppUtils.log("Shortcut key of clipping whose creation is being undone is: `" + key + "'"); }
    gClippingsList.selectedURI = undo.uri;
    pos = undo.pos || gClippingsSvc.indexOf(undo.uri);
    deleteClippingHelper(undo.uri);
    gAppUtils.log(String.ae_fmt("Entry creation has been undone\nEntry name: %S; entry URI: %S", undo.name, undo.uri));

    gRedoStack.push({action: ACTION_CREATENEW, uri: undo.uri, 
 	             name: undo.name, text: undo.text, pos: pos, key: key,
		     parentFolderURI: undo.parentFolderURI});
  }
  else if (undo.action == ACTION_CREATENEWFOLDER) {
    gClippingsList.selectedURI = undo.uri;
    pos = undo.pos || gClippingsSvc.indexOf(undo.uri);
    deleteClippingHelper(undo.uri);
    gRedoStack.push({action: ACTION_CREATENEWFOLDER, name: undo.name,
	             pos: pos, uri: undo.uri,
		     parentFolderURI: undo.parentFolderURI});
    gAppUtils.log("Folder creation undone; folder name: \"" + undo.name + "\"; URI: \"" + undo.uri + "\"");
  }
  else if (undo.action == ACTION_DELETEFOLDER) {
    gClippingsSvc.reattachToFolder(undo.parentFolderURI, undo.uri, undo.pos);
    gIsClippingsDirty = true;

    gClippingsList.selectedURI = undo.uri;
    gCurrentListItemIndex = gClippingsList.selectedIndex;
    updateDisplay();
    removeFolderMenuSeparator();

    gRedoStack.push({action: ACTION_DELETEFOLDER, uri: undo.uri});
    commit();
  }
  else if (undo.action == ACTION_DELETEEMPTYFOLDER) {
    newFolderHelper(undo.parentFolderURI, undo.name, undo.uri, undo.pos, null);
    gCurrentListItemIndex = gClippingsList.selectedIndex;
    updateDisplay();
    
    gRedoStack.push({action: ACTION_DELETEEMPTYFOLDER, uri: undo.uri});
  }
  else if (undo.action == ACTION_MOVETOFOLDER) {
    pos = undo.originalPos;
    try {
      srcFolder = gClippingsSvc.getParent(undo.uri);
      moveToFolderHelper(undo.uri, srcFolder, undo.originalFolder,
			 undo.originalURI, pos, REDO_STACK, true);
    } catch (e) {
      throw ("Exception occurred while attempting to undo item move:\n" + e);
    }
    gAppUtils.log("Move undone.");
  }
  else if (undo.action == ACTION_COPYTOFOLDER) {
    var destFolder = gClippingsSvc.getParent(undo.copyURI);
    deleteClippingHelper(undo.copyURI);
    gRedoStack.push({action:      ACTION_COPYTOFOLDER,
		     originalURI: undo.originalURI,
		     srcFolder:   undo.srcFolder,
		     destFolder:  destFolder,
		     copyURI: undo.copyURI}); 
    gAppUtils.log("Copy undone.");
  }
  else if (undo.action == ACTION_CHANGEPOSITION) {
    moveEntry(undo.uri, undo.parentFolderURI, undo.currPos, undo.prevPos, REDO_STACK);
    gClippingsList.selectedURI = undo.uri;
    updateDisplay();
  }
  else if (undo.action == ACTION_SETSHORTCUTKEY) {
    var clippingKey = $("clipping-key");
    gClippingsList.selectedURI = undo.uri;
    updateDisplay();

    gShortcutKey.setOldKey();
    clippingKey.selectedIndex = undo.keyIdx;
    var oldKey = clippingKey.menupopup.childNodes[undo.keyIdx].label;
    gShortcutKey.update(REDO_STACK);
  }
}


function reverseLastUndo()
{
  updateCurrentClippingData();

  if (gRedoStack.length == 0) {
    gAppUtils.beep();
    return;
  }

  var redo = gRedoStack.pop();
  var srcFolder;  // Used for redoing move or copy
  var pos;        // Used for redoing folder or clipping creation

  if (! gClippingsSvc.exists(redo.uri) && redo.action != ACTION_CREATENEW
      && redo.action != ACTION_CREATENEWFOLDER && redo.action != ACTION_COPYTOFOLDER) {
    gAppUtils.beep();
    return;
  }

  if (redo.action == ACTION_DELETECLIPPING) {
    gClippingsList.selectedURI = redo.uri;
    deleteClippingHelper(redo.uri, UNDO_STACK);
    gAppUtils.log(String.ae_fmt("Entry deletion redone!\nEntry name: %S; entry URI: %S", redo.name, redo.uri));
  }
  else if (redo.action == ACTION_EDITNAME) {
    updateNameHelper(redo.uri, redo.name, UNDO_STACK);
    gClippingsList.selectedURI = redo.uri;
    updateDisplay();

    var clippingName = $("clipping-name");
    clippingName.focus();
    clippingName.select();

    gAppUtils.log(String.ae_fmt("Entry name edit redone!\nEntry name: %S; entry URI:", redo.name, redo.uri));
  }
  else if (redo.action == ACTION_EDITTEXT) {
    updateTextHelper(redo.uri, redo.text, UNDO_STACK);
    gClippingsList.selectedURI = redo.uri;
    updateDisplay();

    var clippingText = $("clipping-text");
    clippingText.focus();
    clippingText.select();

    gAppUtils.log(String.ae_fmt("Entry text edit redone!\nEntry name: %S; entry URI: %S; oldText: %S", redo.name, redo.uri, redo.text));
  }
  else if (redo.action == ACTION_CREATENEW) {
    pos = null;  //redo.pos;
    newClippingHelper(redo.parentFolderURI, redo.name, redo.text, redo.uri, pos, UNDO_STACK);
    gClippingsList.selectedURI = redo.uri;

    gAppUtils.log("Shortcut key of clipping whose creation is being redone (empty if none was defined): `" + redo.key + "'");

    if (redo.key) {
      gClippingsSvc.setShortcutKey(redo.uri, redo.key);
      gIsClippingsDirty = true;
      commit();
      updateDisplay();
    }

    gAppUtils.log(String.ae_fmt("Redoing new clipping creation!\nEntry name: %S; entry URI: %S", redo.name, redo.uri));
  }
  else if (redo.action == ACTION_CREATENEWFOLDER) {
    pos = null;  //redo.pos;
    newFolderHelper(redo.parentFolderURI, redo.name, redo.uri, pos, UNDO_STACK);
    gClippingsList.selectedURI = redo.uri;
    gAppUtils.log("Folder creation redone!  Folder name: \"" + redo.name + "\"");
  }
  else if (redo.action == ACTION_DELETEFOLDER) {
    deleteClippingHelper(redo.uri, UNDO_STACK);
  }
  else if (redo.action == ACTION_DELETEEMPTYFOLDER) {
    deleteClippingHelper(redo.uri, UNDO_STACK);
  }

  else if (redo.action == ACTION_MOVETOFOLDER) {
    try { 
      srcFolder = gClippingsSvc.getParent(redo.uri);
      moveToFolderHelper(redo.uri, srcFolder, redo.originalFolder,
			 redo.originalURI, null, UNDO_STACK, false);
    } catch (e) {
      throw ("Exception occurred while attempting to redo item move:\n" + e);
    }
    gAppUtils.log("Move redone!");
  }
  else if (redo.action == ACTION_COPYTOFOLDER) {
    copyToFolderHelper(redo.originalURI, redo.srcFolder, redo.destFolder,
		       redo.copyURI, null, UNDO_STACK, false);
    gAppUtils.log("Copy redone!");
  }
  else if (redo.action == ACTION_CHANGEPOSITION) {
    moveEntry(redo.uri, redo.parentFolderURI, redo.currPos, redo.prevPos, UNDO_STACK);
    gClippingsList.selectedURI = redo.uri;
    updateDisplay();
  }
  else if (redo.action == ACTION_SETSHORTCUTKEY) {
    var clippingKey = $("clipping-key");
    gClippingsList.selectedURI = redo.uri;
    updateDisplay();

    gShortcutKey.setOldKey();
    clippingKey.selectedIndex = redo.keyIdx;
    var oldKey = clippingKey.menupopup.childNodes[redo.keyIdx].label;
    gShortcutKey.update(UNDO_STACK);
  }
  
  gStatusBar.label = gStrBundle.getString("redoStatus");
}


function showAppInfo()
{
  updateCurrentClippingData();

  window.openDialog("chrome://clippings/content/about.xul", "about_dlg", "centerscreen,dialog,modal");
}


function showShortcutKeyMinihelp()
{
  var helpStr = gStrBundle.getString('shortcutKeyMiniHelp');
  openHelpDialog(gStrBundle.getString("shortcutKeyHlp"), helpStr);
}


function showHelp() 
{
  updateCurrentClippingData();
  openHelpDialog(gStrBundle.getString("clipmanHlp"),
		 gStrBundle.getString('helpMsg'));
}


function openHelpDialog(aHelpTitle, aHelpText)
{
  window.openDialog("chrome://clippings/content/miniHelp.xul", "ae_minihlp_wnd", "centerscreen,dialog,modal", aHelpTitle, aHelpText);
}


