/***************************************************************************
                          kfilereplace.cpp  -  description
                             -------------------
    begin                : sam oct 16 15:28:00 CEST 1999
    copyright            : (C) 1999 by Franois Dupoux
    email                : fdupoux@lemel.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// include files for QT
#include <qdir.h>
#include <qstrlist.h>
#include <qprinter.h>
#include <qpainter.h>
#include <qlistview.h>
#include <qmessagebox.h>
#include <qregexp.h>
#include <qfiledialog.h>

// include files for KDE
#include <kiconloader.h>
#include <kmsgbox.h>
#include <kfiledialog.h>
#include <knewpanner.h>
#include <ktablistbox.h>
#include <kfm.h>

// application specific includes
#include <kfilereplace.h>
#include "kfilereplaceview.h"
#include "kfilereplacedoc.h"
#include "addstringdlg.h"
#include "resource.h"
#include "apistruct.h"
#include "apifiles.h"

// general unix includes
#include <unistd.h>
#include <stdio.h>
//#include <pthread.h>

// Global Thread data
bool g_bThreadRunning = false;
bool g_bThreadMustStop = false;
int g_nFilesRep = 0;
RepDirArg g_argu;
char g_szErrMsg[1024] = "";
//pthread_t g_threadReplace;

// ===========================================================================================================================
KFileReplaceApp::KFileReplaceApp()
{
	config=kapp->getConfig();
	
  ///////////////////////////////////////////////////////////////////
  // call inits to invoke all other construction parts
  initMenuBar();
  initToolBar();
  initStatusBar();
  initKeyAccel();
  initDocument();
  initView();
	
	// Initialize variables
	g_bThreadRunning = false;
	g_bThreadMustStop = false;;
	g_nFilesRep = 0;
	*g_szErrMsg = 0;

  readOptions();
	updateCommands(); // Gray or ungray commands
}

// ===========================================================================================================================
KFileReplaceApp::~KFileReplaceApp()
{

}

// ===========================================================================================================================
void KFileReplaceApp::updateCommands() // Gray or ungray commands
{
  enableCommand(ID_FILE_NEW, !g_bThreadRunning);
  enableCommand(ID_FILE_SEARCH, !g_bThreadRunning);
  enableCommand(ID_FILE_REPLACE, !g_bThreadRunning && view -> getStringsList() -> childCount() > 0);
  enableCommand(ID_FILE_STOP, g_bThreadRunning);
  enableCommand(ID_FILE_SAVE, !g_bThreadRunning && view -> getResultList() -> childCount() > 0);

  enableCommand(ID_STRINGS_ADD, !g_bThreadRunning);
  enableCommand(ID_STRINGS_DEL, !g_bThreadRunning && view -> getStringsList() -> childCount() > 0);
  enableCommand(ID_STRINGS_EMPTY, !g_bThreadRunning && view -> getStringsList() -> childCount() > 0);
  enableCommand(ID_STRINGS_EDIT, !g_bThreadRunning && view -> getStringsList() -> childCount() > 0);
  enableCommand(ID_STRINGS_SAVE, !g_bThreadRunning && view -> getStringsList() -> childCount() > 0);
  enableCommand(ID_STRINGS_LOAD, !g_bThreadRunning);
  enableCommand(ID_STRINGS_INVERTCUR, !g_bThreadRunning && view -> getStringsList() -> childCount() > 0);
  enableCommand(ID_STRINGS_INVERTALL, !g_bThreadRunning && view -> getStringsList() -> childCount() > 0);


  enableCommand(ID_OPTIONS_RECURSIVE, !g_bThreadRunning);
  enableCommand(ID_OPTIONS_BAKUP, !g_bThreadRunning);
  enableCommand(ID_OPTIONS_CASESENSITIVE, !g_bThreadRunning);

  enableCommand(ID_RESULT_INFO, !g_bThreadRunning && view -> getResultList() -> childCount() > 0);
  enableCommand(ID_RESULT_OPEN, !g_bThreadRunning && view -> getResultList() -> childCount() > 0);
  enableCommand(ID_RESULT_DIROPEN, !g_bThreadRunning && view -> getResultList() -> childCount() > 0);
  enableCommand(ID_RESULT_DELETE, !g_bThreadRunning && view -> getResultList() -> childCount() > 0);

	// Redraw Open recent menu
  recentStringPopup -> clear();
  for (int i = 0; i < (int) recentStringFileList.count(); i++)
    recentStringPopup -> insertItem(recentStringFileList.at(i));

}

// ===========================================================================================================================
void KFileReplaceApp::initKeyAccel()
{
	key_accel = new KAccel( this );
	
	// file_menu accelerators
	key_accel->connectItem( KAccel::New, this, SLOT( slotFileNew() ) );
	key_accel->connectItem( KAccel::Quit, this, SLOT( slotFileQuit() ) );
	// edit_menu accelerators
	key_accel->changeMenuAccel(	file_menu, ID_FILE_NEW, KAccel::New );
	key_accel->changeMenuAccel(	file_menu, ID_FILE_QUIT, KAccel::Quit );

  key_accel->readSettings();	
}

// ===========================================================================================================================
void KFileReplaceApp::initMenuBar()
{

  ///////////////////////////////////////////////////////////////////
  // MENUBAR

  ///////////////////////////////////////////////////////////////////
  // menuBar entry file_menu
  file_menu = new QPopupMenu();
  file_menu->insertItem(kapp->getMiniIcon(), i18n("New &Window"), ID_FILE_NEW_WINDOW );
  file_menu->insertSeparator();
  file_menu->insertItem(Icon("filenew.xpm"), i18n("&New"), ID_FILE_NEW );
  file_menu->insertItem(Icon("filesearch.xpm"), i18n("&Search"), ID_FILE_SEARCH );
  file_menu->insertItem(Icon("filereplace.xpm"), i18n("&Replace"), ID_FILE_REPLACE );
  file_menu->insertItem(Icon("filestop.xpm"), i18n("Sto&p"), ID_FILE_STOP );
  file_menu->insertSeparator();
  file_menu->insertItem(Icon("filesave.xpm"), i18n("Save &results"), ID_FILE_SAVE );

  file_menu->insertSeparator();
  file_menu->insertItem(i18n("E&xit"), ID_FILE_QUIT );

	// *-*-*-*-*-*-*-*-*-*- Recent File List *-*-*-*-*--*-*-*-*-*-*-*-*-*-
  recentStringPopup = new QPopupMenu;
  recentStringPopup -> clear();
  for (int i = 0; i < (int) recentStringFileList.count(); i++)
    recentStringPopup -> insertItem(recentStringFileList.at(i));

	// *-*-*-*-*-*-*-*-*-*- Recent File List *-*-*-*-*--*-*-*-*-*-*-*-*-*-
	
  ///////////////////////////////////////////////////////////////////
  // menuBar entry strings_menu
  strings_menu = new QPopupMenu();
  strings_menu->setCheckable(true);
  strings_menu->insertItem(Icon("stradd.xpm"), i18n("&Add string"), ID_STRINGS_ADD);
  strings_menu->insertItem(Icon("strdel.xpm"), i18n("&Delete selected string"), ID_STRINGS_DEL );
  strings_menu->insertItem(Icon("strempty.xpm"), i18n("&Empty string list"), ID_STRINGS_EMPTY );
  strings_menu->insertItem(Icon("stredit.xpm"), i18n("Edi&t selected string"), ID_STRINGS_EDIT );
  strings_menu->insertSeparator();
  strings_menu->insertItem(Icon("strsave.xpm"), i18n("&Save strings list to file..."), ID_STRINGS_SAVE );
  strings_menu->insertItem(Icon("strload.xpm"), i18n("&Load strings list from file..."), ID_STRINGS_LOAD );
  strings_menu->insertItem(Icon("strload.xpm"), i18n("&Load recent strings files..."), recentStringPopup);
  strings_menu->insertSeparator();
  strings_menu->insertItem(Icon("strinvert.xpm"), i18n("&Invert current string (search <--> replace)"), ID_STRINGS_INVERTCUR );
  strings_menu->insertItem(Icon("strinvert.xpm"), i18n("&Invert all strings (search <--> replace)"), ID_STRINGS_INVERTALL );

  ///////////////////////////////////////////////////////////////////
  // menuBar entry options_menu
  options_menu = new QPopupMenu();
  options_menu->setCheckable(true);
  options_menu->insertItem(i18n("&Include sub-folders"), ID_OPTIONS_RECURSIVE);
  options_menu->insertItem(i18n("Crete &bakup"), ID_OPTIONS_BAKUP );
  options_menu->insertItem(i18n("Case &sensitive"), ID_OPTIONS_CASESENSITIVE );

  ///////////////////////////////////////////////////////////////////
  // menuBar entry results_menu
  results_menu = new QPopupMenu();
  results_menu->insertItem(Icon("resfileinfo.xpm"), i18n("&Informations about file..."), ID_RESULT_INFO);
  results_menu->insertItem(Icon("resfileopen.xpm"), i18n("&Open file"), ID_RESULT_OPEN );
  results_menu->insertItem(Icon("resdiropen.xpm"), i18n("Open &Directory"), ID_RESULT_DIROPEN );
  results_menu->insertItem(Icon("resfiledel.xpm"), i18n("&Delete file"), ID_RESULT_DELETE );

  ///////////////////////////////////////////////////////////////////
  // menuBar entry view_menu
  view_menu = new QPopupMenu();
  view_menu->setCheckable(true);
  view_menu->insertItem(i18n("&Toolbar"), ID_VIEW_TOOLBAR);
  view_menu->insertItem(i18n("&Statusbar"), ID_VIEW_STATUSBAR );

  ///////////////////////////////////////////////////////////////////
  // menuBar entry help_menu
  help_menu = new QPopupMenu();
  //help_menu = kapp->getHelpMenu(true, i18n(IDS_APP_ABOUT));
  help_menu->insertItem(i18n("KFileReplace &Home page"), ID_HELP_WEBSITE);
  help_menu->insertItem(i18n("&About KFileReplace..."), ID_HELP_ABOUT);

  ///////////////////////////////////////////////////////////////////
  // MENUBAR CONFIGURATION
  // insert your popup menus with the according menubar entries in the order
  // they will appear later from left to right
  menuBar()->insertItem(i18n("&File"), file_menu);
  menuBar()->insertItem(i18n("&Strings"), strings_menu);
  menuBar()->insertItem(i18n("&Options"), options_menu);
  menuBar()->insertItem(i18n("&Results"), results_menu);
  menuBar()->insertItem(i18n("&View"), view_menu);

  menuBar()->insertSeparator();
  menuBar()->insertItem(i18n("&Help"), help_menu);

  ///////////////////////////////////////////////////////////////////
  // CONNECT THE MENU SLOTS WITH SIGNALS
  // for execution slots and statusbar messages
  connect( recentStringPopup, SIGNAL(activated(int)), SLOT(slotOpenRecentStringFile(int)));

  connect(file_menu,SIGNAL(activated(int)),SLOT(commandCallback(int)));
  connect(file_menu,SIGNAL(highlighted(int)),SLOT(statusCallback(int)));

  connect(strings_menu,SIGNAL(activated(int)),SLOT(commandCallback(int)));
  connect(strings_menu,SIGNAL(highlighted(int)),SLOT(statusCallback(int)));

  connect(options_menu,SIGNAL(activated(int)),SLOT(commandCallback(int)));
  connect(options_menu,SIGNAL(highlighted(int)),SLOT(statusCallback(int)));

  connect(results_menu,SIGNAL(activated(int)),SLOT(commandCallback(int)));
  connect(results_menu,SIGNAL(highlighted(int)),SLOT(statusCallback(int)));

  connect(view_menu,SIGNAL(activated(int)),SLOT(commandCallback(int)));
  connect(view_menu,SIGNAL(highlighted(int)),SLOT(statusCallback(int)));

  connect(help_menu,SIGNAL(activated(int)),SLOT(commandCallback(int)));
  connect(help_menu,SIGNAL(highlighted(int)),SLOT(statusCallback(int)));

}

// ===========================================================================================================================
void KFileReplaceApp::initToolBar()
{

  ///////////////////////////////////////////////////////////////////
  // TOOLBAR
  toolBar()->insertButton(Icon("filenew.xpm"), ID_FILE_NEW, true, i18n("New operation") );
  toolBar()->insertButton(Icon("filesearch.xpm"), ID_FILE_SEARCH, true, i18n("Search") );
  toolBar()->insertButton(Icon("filereplace.xpm"), ID_FILE_REPLACE, true, i18n("Replace") );
  toolBar()->insertButton(Icon("filestop.xpm"), ID_FILE_STOP, true, i18n("Stop operation") );
  toolBar()->insertSeparator();
  toolBar()->insertButton(Icon("stradd.xpm"), ID_STRINGS_ADD, true, i18n("Add string") );
  toolBar()->insertButton(Icon("strdel.xpm"), ID_STRINGS_DEL, true, i18n("Delete selected string") );
  toolBar()->insertButton(Icon("stredit.xpm"), ID_STRINGS_EDIT, true, i18n("Edit selected string") );
  toolBar()->insertButton(Icon("strload.xpm"), ID_STRINGS_LOAD, true, i18n("&Load strings list from file...") );
  toolBar()->setDelayedPopup(ID_STRINGS_LOAD, recentStringPopup);
  toolBar()->insertSeparator();
  toolBar()->insertButton(Icon("filesave.xpm"), ID_FILE_SAVE, true, i18n("Save operation results"));
  toolBar()->insertSeparator();
  toolBar()->insertButton(Icon("resfileinfo.xpm"), ID_RESULT_INFO, true, i18n("Informations about selected file"));
  toolBar()->insertButton(Icon("resfileopen.xpm"), ID_RESULT_OPEN, true, i18n("Open selected file"));
  toolBar()->insertButton(Icon("resdiropen.xpm"), ID_RESULT_DIROPEN, true, i18n("Open parent directory"));
  toolBar()->insertButton(Icon("resfiledel.xpm"), ID_RESULT_DELETE, true, i18n("Delete selected file"));
  toolBar()->insertSeparator();
  toolBar()->insertButton(Icon("help.xpm"), ID_HELP_CONTENTS, SIGNAL(clicked() ),
  			kapp, SLOT( appHelpActivated() ), true,i18n("Help"));

  ///////////////////////////////////////////////////////////////////
  // INSERT YOUR APPLICATION SPECIFIC TOOLBARS HERE WITH toolBar(n)


  ///////////////////////////////////////////////////////////////////
  // CONNECT THE TOOLBAR SLOTS WITH SIGNALS - add new created toolbars by their according number
	// connect for invoking the slot actions
  connect(toolBar(), SIGNAL(clicked(int)), SLOT(commandCallback(int)));
	// connect for the status help on holing icons pressed with the mouse button
  connect(toolBar(), SIGNAL(pressed(int)), SLOT(statusCallback(int)));


}

// ===========================================================================================================================
void KFileReplaceApp::initStatusBar()
{
  ///////////////////////////////////////////////////////////////////
  // STATUSBAR
	// TODO: add your own items you need for displaying current application status.
  statusBar()->insertItem(i18n("Ready."), ID_STATUS_MSG );
}

// ===========================================================================================================================
void KFileReplaceApp::initDocument()
{
  int nRes;

	doc = new KFileReplaceDoc(this);
  nRes = doc->newDocument();
	if (nRes == false)
		slotFileQuit();	
}

// ===========================================================================================================================
void KFileReplaceApp::initView()
{
  ////////////////////////////////////////////////////////////////////
  // create the main widget here that is managed by KTMainWindow's view-region and
  // connect the widget to your document to display document contents.

  view = new KFileReplaceView(this,"view",KNewPanner::Horizontal,KNewPanner::Percent,50);

  doc->addView(view);
  setView(view);

	view -> init();

  QString caption=kapp->getCaption();	
  setCaption(caption+": "+doc->getTitle());

}

// ===========================================================================================================================
void KFileReplaceApp::enableCommand(int id_, bool bState)
{
  ///////////////////////////////////////////////////////////////////
  // enable menu and toolbar functions by their ID's
  menuBar()->setItemEnabled(id_,bState);
  toolBar()->setItemEnabled(id_,bState);
}


// ===========================================================================================================================
int KFileReplaceApp::checkBeforeOperation(int nTypeOfOperation)
{
	// ================= Check Thread is not running ========================
	if (g_bThreadRunning)
	{	KMsgBox::message(this, "KFileReplace", i18n("The replacing operation is already running. You must finish it before."), KMsgBox::EXCLAMATION, i18n("Cancel"));
		return -1;
	}

	// Check there are strings to replace (not need in search operation)
  if (nTypeOfOperation == OPERATION_REPLACE && view -> getStringsList() -> childCount() == 0)
	{	KMsgBox::message(this, "KFileReplace", i18n("There are no strings to search and replace."), KMsgBox::EXCLAMATION, i18n("Cancel"));	
		return -1;
	}

 	 // =========================================== OPTIONS ==================================================
	bool bRecursive, bCaseSensitive, bBakup;
  bRecursive = options_menu -> isItemChecked(ID_OPTIONS_RECURSIVE);
  bCaseSensitive = options_menu -> isItemChecked(ID_OPTIONS_CASESENSITIVE);
  bBakup = options_menu -> isItemChecked(ID_OPTIONS_BAKUP);
	
	// ================== Prepare argument structure to pass to the ReplaceDirectory function ========================
	
	sprintf (g_argu.szDir, "%s", doc -> m_szProjectDirectory);
	sprintf (g_argu.szFilter, "%s", doc -> m_szProjectFilter);
	g_argu.qlvResult = view -> getResultList();
	g_argu.qlvStrings = view -> getStringsList();
	g_argu.bMinSize = doc -> m_bMinSize;
	g_argu.bMaxSize = doc -> m_bMaxSize;
	g_argu.nMinSize = doc -> m_nMinSize;
	g_argu.nMaxSize = doc -> m_nMaxSize;
	g_argu.nTypeOfAccess = doc -> m_nTypeOfAccess;
	g_argu.bMinDate = doc -> m_bMinDate;
	g_argu.bMaxDate = doc -> m_bMaxDate;
	g_argu.qdMinDate = doc -> m_qdMinDate;
	g_argu.qdMaxDate = doc -> m_qdMaxDate;
	g_argu.bCaseSensitive = bCaseSensitive;
	g_argu.bRecursive = bRecursive;
	g_argu.bBakup = bBakup;
	g_argu.app = (QWidget *) this;
	g_argu.view = view;

	// Clear the list view
	view -> getResultList() -> clear();

	return 0; // Operation successfully prepared
}

// ===========================================================================================================================
void KFileReplaceApp::slotFileSearch()
{
	if (checkBeforeOperation(OPERATION_SEARCH) == -1)
		return;
	
	slotStatusMsg(i18n("Searching files..."));
	
	// Run the Replacing operation Thread
	g_bThreadRunning = true;
	g_bThreadMustStop = false;
	g_nFilesRep = 0;

	updateCommands();
	
	// Default error message
	*g_szErrMsg = 0;

	//nRes = pthread_create(&g_threadReplace, NULL, SearchThread, (void *) &g_argu);
	startTimer(100);
	SearchThread( (void *) &g_argu );

	/*if (nRes != 0) // Can't create Thread	
	{	slotStatusMsg(i18n("Can't create thread to run the replace operation."));
		return ;
	}
	else // if success
	{	startTimer(100);
	}*/

}


// ===========================================================================================================================
void KFileReplaceApp::slotFileReplace()
{
	if (checkBeforeOperation(OPERATION_REPLACE) == -1)
		return;


	slotStatusMsg(i18n("Replacing files..."));
	
	// Run the Replacing operation Thread
	g_bThreadRunning = true;
	g_bThreadMustStop = false;
	g_nFilesRep = 0;

	updateCommands();
	
	// Default error message
	*g_szErrMsg = 0;

	//nRes = pthread_create(&g_threadReplace, NULL, ReplaceThread, (void *) &g_argu);
	startTimer(100);
	ReplaceThread((void *) &g_argu);

	/*if (nRes != 0) // Can't create Thread	
	{	slotStatusMsg(i18n("Can't create thread to run the replace operation."));
		return ;
	}
	else // if success
	{	startTimer(100);
	} */
}

// ===========================================================================================================================
void KFileReplaceApp::slotFileStop()
{
	if (g_bThreadRunning == false) // If no thread running
		return ;

	// Stop the current thread
	g_bThreadMustStop = true;

	//pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);
	//pthread_join(&g_threadReplace, NULL); // Wait the thread finished
}

// ===========================================================================================================================
void KFileReplaceApp::slotFileSave()
{	QFileDialog dlg;
	QString strFilename;
	char szMess[MAX_MESSAGE];
	QListView *lvResult;

	lvResult = view -> getResultList();
	
	// Check there are results
	if (lvResult -> childCount() == 0)
	{	KMsgBox::message(this, "KFileReplace", i18n("There are no results to save: the result list is empty."), KMsgBox::EXCLAMATION, i18n("Cancel"));	
   	return ;
	}

	// Select the file where results will be saved
	strFilename = dlg.getSaveFileName(0, "*.htm");
	
	if (strFilename.isEmpty())
		return ;

	// Force the extension to be "htm" or "html"
	if((strFilename.size() > 3 && strFilename.right(4) != ".htm") && (strFilename.size() > 4 && strFilename.right(5) != ".html"))
		strFilename += ".html";

	// Save results to file
	
  // a) Open the file
	FILE *fResults;
	fResults = fopen(strFilename.data(),"w");

	if (fResults == NULL) // Can't open file
	{	sprintf (szMess, i18n("Can't open the file %s in writing mode to save results."), strFilename.data() );
		KMsgBox::message(this, "KFileReplace", szMess, KMsgBox::EXCLAMATION, i18n("Cancel"));	
   	return ;
	}

	// b) Write header of the HTML file
	fprintf(fResults, "<HTML><HEAD>\n");
	fprintf(fResults, "<!DOCTYPE %s>\n", i18n("KFileReplace Results File"));
	fprintf(fResults, "<TITLE>%sKFileReplace Results File</TITLE></HEAD>\n", i18n("KFileReplace Results File"));
	fprintf(fResults, "<BODY><H1>%s</H1>\n", i18n("KFileReplace Results File"));
	fprintf(fResults, "<DL><p>\n");

	// c) Write the file list
	QListViewItem *lviCurItem;
	QListViewItem *lviFirst;
	QString strPath;
	
	lviCurItem = lviFirst = lvResult -> firstChild();
	if (lviCurItem == NULL)
	{	return ;
	}

	do
	{	strPath.sprintf("%s/%s", lviCurItem -> text(1), lviCurItem -> text(0));
		fprintf(fResults,"<DT><A HREF=\"file:%s\">file:%s</A><br>\n",	strPath.data(), strPath.data());
		fprintf(fResults,i18n("Size: %s --> %s **** %.3ld strings replaced.<br><br>\n"), lviCurItem -> text(2), lviCurItem -> text(3), atol(lviCurItem -> text(2)));

    lviCurItem = lviCurItem -> nextSibling();
	} while(lviCurItem && lviCurItem != lviFirst);


	// d) Write the end of the file
	fprintf(fResults,"</DL><P></BODY></HTML>\n");
	fclose(fResults);
}

// ===========================================================================================================================
void KFileReplaceApp::timerEvent(QTimerEvent *)
{
	updateCommands();


	if (g_bThreadRunning == false) // Thread not running: it has finished since the last call of this function
	{	killTimers();

		sprintf(g_szErrMsg, "%ld files successfully found/replaced", g_nFilesRep);				
		slotStatusMsg(g_szErrMsg); // Draw the error or success message
	}
}

// ===========================================================================================================================
void KFileReplaceApp::slotStringsAdd()
{	view -> slotStringAdd();
  return;

}

// ===========================================================================================================================
void KFileReplaceApp::slotStringsDel()
{	delete view -> getStringsList() -> currentItem(); // Remove item from ListView
}

// ===========================================================================================================================
void KFileReplaceApp::slotStringsEmpty()
{	view -> getStringsList() -> clear();
}

// ===========================================================================================================================
void KFileReplaceApp::slotStringsEdit()
{	
	QListViewItem *lviCurItem;
	lviCurItem = view -> getStringsList() -> currentItem();	
	if (lviCurItem == 0)
		return;

	AddStringDlg dlg(this);
	
	dlg.setSearchText( lviCurItem -> text(0) );
	dlg.setReplaceText( lviCurItem -> text(1) );

	if (dlg.exec() == QDialog::Rejected) // If Cancel
		return ;

  lviCurItem -> setText(0, dlg.m_strSearch.data());
  lviCurItem -> setText(1, dlg.m_strReplace.data());
}

// ===========================================================================================================================
void KFileReplaceApp::slotStringsSave()
{	
	QFileDialog dlg;
	QString strFilename;
	char szMess[MAX_MESSAGE];
	QListViewItem *lviCurItem;
	QListViewItem *lviFirst;
	int nItemPos;

	// Check there are strings in the list
	if (view -> getStringsList() -> childCount() == 0)
	{	KMsgBox::message(this, "KFileReplace", i18n("There are no strings to save in the list."), KMsgBox::EXCLAMATION, i18n("Cancel"));	
   	return ;
	}

	// Select the file where strings will be saved	
	strFilename = dlg.getSaveFileName(0, "*.kfr");
	if (strFilename.isEmpty())
		return;

	// Force the extension to be "kfr" == KFileReplace extension
	if(strFilename.size() > 3)
	{	QString strExt = strFilename.right(4);
		if (strExt != ".kfr" && strExt.right(1) != ".")
			strFilename += ".kfr";
	}

  // a) Open the file
	FILE *fStrings;
	fStrings = fopen(strFilename.data(),"w");

	if (fStrings == NULL) // Can't open file
	{	sprintf (szMess, i18n("Can't open the file %s in writing mode to save strings list."), strFilename.data() );
		KMsgBox::message(this, "KFileReplace", szMess, KMsgBox::EXCLAMATION, i18n("Cancel"));	
   	return ;
	}

	// b) Write header of the KFR file
	KFRHeader head;
	memset(&head, 0, sizeof(KFRHeader));

	sprintf(head.szPgm, "KFileReplace");
  head.nStringsNb = view -> getStringsList() -> childCount();

	if (fwrite(&head, sizeof(KFRHeader), 1, fStrings) != 1)
	{	sprintf (szMess, i18n("Can't write data in file %s"), strFilename.data() );
		KMsgBox::message(this, "KFileReplace", szMess, KMsgBox::EXCLAMATION, i18n("Cancel"));	
		fclose(fStrings);
   	return ;
	}

	// c) Write the file strings
	int nOldTextSize, nNewTextSize;
	int nErrors = 0;

	nItemPos = 0;
	lviCurItem = lviFirst = view -> getStringsList() -> firstChild();
	if (lviCurItem == NULL)
		return ;

	do
	{	nOldTextSize = strlen(lviCurItem -> text(0));
		nNewTextSize = strlen(lviCurItem -> text(1));
    nErrors += (fwrite(&nOldTextSize, sizeof(int), 1, fStrings)) != 1;
    nErrors += (fwrite(&nNewTextSize, sizeof(int), 1, fStrings)) != 1;
    nErrors += (fwrite(lviCurItem -> text(0), nOldTextSize, 1, fStrings)) != 1;
    if (nNewTextSize > 0)
			nErrors += (fwrite(lviCurItem -> text(1), nNewTextSize, 1, fStrings)) != 1;
		if (nErrors > 0)
		{	sprintf (szMess, i18n("Can't write data in file %s"), strFilename.data() );
			KMsgBox::message(this, "KFileReplace", szMess, KMsgBox::EXCLAMATION, i18n("Cancel"));	
			fclose(fStrings);
   		return ;
		}
		lviCurItem = lviCurItem -> nextSibling();
	} while(lviCurItem && lviCurItem != lviFirst);


	// d) Close the file
	fclose(fStrings);
}


// ===========================================================================================================================
void KFileReplaceApp::slotOpenRecentStringFile(int i)
{
	QString strFilename;

	// Select the file to load from
	strFilename = recentStringFileList.at(i);

	LoadStringFile(strFilename);
}

// ===========================================================================================================================
void KFileReplaceApp::LoadStringFile(QString strFilename)
{
	char szMess[MAX_MESSAGE];
	QListViewItem *lvi;

	if (strFilename.isEmpty())
		return;

  // a) Open the file
	FILE *fStrings;
	fStrings = fopen(strFilename.data(),"r");

	if (fStrings == NULL) // Can't open file
	{	sprintf (szMess, i18n("Can't open the file %s in reading mode to load trings list."), strFilename.data() );
		KMsgBox::message(this, "KFileReplace", szMess, KMsgBox::EXCLAMATION, i18n("Cancel"));	
   	return ;
	}

	// b) Read header of the KFR file
	KFRHeader head;

	if (fread(&head, sizeof(KFRHeader), 1, fStrings) != 1)
	{	sprintf (szMess, i18n("Can't read data from file %s"), strFilename.data() );
		KMsgBox::message(this, "KFileReplace", szMess, KMsgBox::EXCLAMATION, i18n("Cancel"));	
		fclose(fStrings);
   	return ;
	}

	// Check the file is a KFileReplace file
	if (strcmp(head.szPgm, "KFileReplace") != 0)
	{	sprintf (szMess, i18n("%s is not a KFileReplace file. Can't continue."), strFilename.data() );
		KMsgBox::message(this, "KFileReplace", szMess, KMsgBox::EXCLAMATION, i18n("Cancel"));	
		fclose(fStrings);
   	return ;
	}

	// Add file to "load strings form file" menu
	AddRecentStringFileToList(strFilename);

	// c) Read the file strings
	int nOldTextSize, nNewTextSize;
	int nErrors = 0;
	char *szString; // Dynamic memory alloc
	int nStrSize;

	// Empty the current list
	view -> getStringsList() -> clear();

	for (int i=0; i < head.nStringsNb; i++)
	{	
    nErrors += (fread(&nOldTextSize, sizeof(int), 1, fStrings)) != 1;
    nErrors += (fread(&nNewTextSize, sizeof(int), 1, fStrings)) != 1;
		if (nErrors > 0)
		{	sprintf (szMess, i18n("Can't read data from file %s"), strFilename.data() );
			KMsgBox::message(this, "KFileReplace", szMess, KMsgBox::EXCLAMATION, i18n("Cancel"));	
			fclose(fStrings);
   		return ;
		}

		// Allocate new string
		nStrSize = ((nOldTextSize > nNewTextSize) ? nOldTextSize : nNewTextSize) + 2;
		szString = new char[nStrSize+10];
    memset(szString, 0, nStrSize);
		if (szString == 0)
		{	KMsgBox::message(this, "KFileReplace", i18n("Out of memory"), KMsgBox::EXCLAMATION, i18n("Cancel"));	
			fclose(fStrings);
   		return ;
		}

		// Read "Search" text
		if (fread(szString, nOldTextSize, 1, fStrings) != 1)
		{	sprintf (szMess, i18n("Can't read data from file %s"), strFilename.data() );
			KMsgBox::message(this, "KFileReplace", szMess, KMsgBox::EXCLAMATION, i18n("Cancel"));	
			fclose(fStrings);
   		return ;
		}

		// Add item to list
		lvi = new QListViewItem(view -> getStringsList(), szString);
  	CHECK_PTR( lvi );
		lvi -> setPixmap(0, view -> getIconString());

		// Read "Replace" text
    memset(szString, 0, nStrSize);
    if (nNewTextSize > 0) // If there is a Replace text
		{	if (fread(szString, nNewTextSize, 1, fStrings) != 1)
			{	printf (szMess, i18n("Can't read data from file %s"), strFilename.data() );
				KMsgBox::message(this, "KFileReplace", szMess, KMsgBox::EXCLAMATION, i18n("Cancel"));	
				fclose(fStrings);
   			return ;
			}
			lvi -> setText(1, szString);
		}

		delete szString;	
	}		

	// d) Close the file
	fclose(fStrings);

}

// ===========================================================================================================================
void KFileReplaceApp::slotStringsLoad()
{	
	QFileDialog dlg;
	QString strFilename;

	// Select the file to load from
	strFilename = dlg.getOpenFileName(0, "*.kfr");

	LoadStringFile(strFilename);
}

// ===========================================================================================================================
void KFileReplaceApp::slotStringsInvertCur()
{	
	QListViewItem *lviCurItem;
	lviCurItem = view -> getStringsList() -> currentItem();	
	if (lviCurItem == 0)
		return;

	QString strSearch, strReplace;
	strSearch = lviCurItem -> text(0);
	strReplace = lviCurItem -> text(1);
  lviCurItem -> setText(0, strReplace.data());
  lviCurItem -> setText(1, strSearch.data());
}

// ===========================================================================================================================
void KFileReplaceApp::slotStringsInvertAll()
{	
	QListViewItem *lviCurItem;
	QListViewItem *lviFirst;
	int nItemPos;
	QString strSearch, strReplace;

	nItemPos = 0;
	lviCurItem = lviFirst = view -> getStringsList() -> firstChild();
	if (lviCurItem == NULL)
		return ;

	do
	{	strSearch = lviCurItem -> text(0);
		strReplace = lviCurItem -> text(1);
    lviCurItem -> setText(0, strReplace.data());
    lviCurItem -> setText(1, strSearch.data());

		lviCurItem = lviCurItem -> nextSibling();
	} while(lviCurItem && lviCurItem != lviFirst);
}

// ===========================================================================================================================
void KFileReplaceApp::slotOptionsRecursive()
{
	if( options_menu -> isItemChecked(ID_OPTIONS_RECURSIVE))
	  options_menu->setItemChecked(ID_OPTIONS_RECURSIVE, false);
	else
		options_menu->setItemChecked(ID_OPTIONS_RECURSIVE, true);

  slotStatusMsg(i18n("Ready."));
}

// ===========================================================================================================================
void KFileReplaceApp::slotOptionsBakup()
{
	if( options_menu -> isItemChecked(ID_OPTIONS_BAKUP))
	  options_menu->setItemChecked(ID_OPTIONS_BAKUP, false);
	else
		options_menu->setItemChecked(ID_OPTIONS_BAKUP, true);

  slotStatusMsg(i18n("Ready."));
}

// ===========================================================================================================================
void KFileReplaceApp::slotOptionsCaseSensitive()
{
	if( options_menu -> isItemChecked(ID_OPTIONS_CASESENSITIVE))
	  options_menu->setItemChecked(ID_OPTIONS_CASESENSITIVE, false);
	else
		options_menu->setItemChecked(ID_OPTIONS_CASESENSITIVE, true);

  slotStatusMsg(i18n("Ready."));
}

// ===========================================================================================================================
void KFileReplaceApp::slotResultInfo()
{
	if (view -> getResultList() -> currentItem() == 0) // If no selected item
		return ;

	QString strFilename;
	KFM *kfm= new KFM();

	strFilename.append(view -> getResultList() -> currentItem() -> text(1)); // Append Directory
	strFilename.append("/"); // Append separator
	strFilename.append(view -> getResultList() -> currentItem() -> text(0)); // Append Filename

	kfm->openProperties(strFilename.data());
	delete kfm;
}

// ===========================================================================================================================
void KFileReplaceApp::slotResultOpen()
{
	if (view -> getResultList() -> currentItem() == 0) // If no selected item
		return ;

	QString tmp= "file:";
	KFM *kfm= new KFM();

	tmp.append(view -> getResultList() -> currentItem() -> text(1)); // Append Directory
	tmp.append("/"); // Append separator
	tmp.append(view -> getResultList() -> currentItem() -> text(0)); // Append Filename
	kfm->exec(tmp.data(),0L);

	delete kfm;
}

// ===========================================================================================================================
void KFileReplaceApp::slotResultDirOpen()
{
	QString tmp= "file:";

	if (view -> getResultList() -> currentItem() == NULL) // If no selected item
		return ;

	KFM *kfm= new KFM();
	tmp.append(view -> getResultList() -> currentItem() -> text(1)); // Append Directory
	kfm->exec(tmp.data(),0L);

	delete kfm;
}

// ===========================================================================================================================
void KFileReplaceApp::slotResultDelete()
{
	QString strFilename;
	char szMess[MAX_MESSAGE];
  QFile qf;

	if (view -> getResultList() -> currentItem() == NULL) // If no selected item
		return ;

	strFilename.append(view -> getResultList() -> currentItem() -> text(1)); // Append Directory
	strFilename.append("/"); // Append separator
	strFilename.append(view -> getResultList() -> currentItem() -> text(0)); // Append Filename
	
  // The user must confirm the deletion of the file
  sprintf (szMess, "Are you sure you really want to delete the file %s ?", strFilename.data());
	if (KMsgBox::yesNo(this, "KFileReplace", szMess, KMsgBox::QUESTION) == 2) // No clicked
		return ; // Cancel

	qf.setName(strFilename.data());
	qf.remove();
	delete view -> getResultList() -> currentItem(); // Remove item from ListView
}

// ===========================================================================================================================
KFileReplaceDoc* KFileReplaceApp::getDocument() const
{
	return doc;
}

// ===========================================================================================================================
void KFileReplaceApp::saveOptions()
{	
	//fprintf(stderr,"Writing options\n");

	config->setGroup("General Options");
	config->writeEntry("Geometry", size() );
  config->writeEntry("Show Toolbar", toolBar()->isVisible());
  config->writeEntry("Show Statusbar",statusBar()->isVisible());
  config->writeEntry("MenuBarPos", (int)menuBar()->menuBarPos());
  config->writeEntry("ToolBarPos",  (int)toolBar()->barPos());


	// options seetings (recursivity, bakup, case sensitive)
	config->setGroup("Options");

  config->writeEntry("Recursivity",  options_menu->isItemChecked(ID_OPTIONS_RECURSIVE) == true);
  config->writeEntry("Bakup",  options_menu->isItemChecked(ID_OPTIONS_BAKUP) == true);
  config->writeEntry("Case sensitive",  options_menu->isItemChecked(ID_OPTIONS_CASESENSITIVE) == true);

	// Recent file list
	QString recent_number;
  config->setGroup("Recently Opened StringsFiles");
  for(int i = 0; i < (int) recentStringFileList.count(); i++)
	{	recent_number.setNum(i+1);
		config->writeEntry(recent_number.data(),recentStringFileList.at(i));
  }

}

// ===========================================================================================================================
void KFileReplaceApp::readOptions()
{
	//fprintf(stderr,"Reading options\n");
	
	config->setGroup("General Options");

	// bar status settings
	bool bViewToolbar = config->readBoolEntry("Show Toolbar", true);
	view_menu->setItemChecked(ID_VIEW_TOOLBAR, bViewToolbar);
	if(!bViewToolbar)
		enableToolBar(KToolBar::Hide);
	
  bool bViewStatusbar = config->readBoolEntry("Show Statusbar", true);
	view_menu->setItemChecked(ID_VIEW_STATUSBAR, bViewStatusbar);
	if(!bViewStatusbar)
		enableStatusBar(KStatusBar::Hide);
	
	// bar position settings
	KMenuBar::menuPosition menu_bar_pos;
	menu_bar_pos=(KMenuBar::menuPosition)config->readNumEntry("MenuBarPos", KMenuBar::Top);

  KToolBar::BarPosition tool_bar_pos;
  tool_bar_pos=(KToolBar::BarPosition)config->readNumEntry("ToolBarPos", KToolBar::Top);

	menuBar()->setMenuBarPos(menu_bar_pos);
  toolBar()->setBarPos(tool_bar_pos);
		
  QSize size=config->readSizeEntry("Geometry");
	if(!size.isEmpty())
		resize(size);
	
	// options seetings (recursivity, bakup, case sensitive)
  bool bState;
	config->setGroup("Options");
	
	bState = config->readBoolEntry("Recursivity", true);
	options_menu->setItemChecked(ID_OPTIONS_RECURSIVE, bState);

	bState = config->readBoolEntry("Bakup", true);
	options_menu->setItemChecked(ID_OPTIONS_BAKUP, bState);

	bState = config->readBoolEntry("Case sensitive", false);
	options_menu->setItemChecked(ID_OPTIONS_CASESENSITIVE, bState);


	// Save Recent Open File List
	QString str;
  recentStringFileList.clear();
	QString recent_number;
  config->setGroup("Recently Opened StringsFiles");

  for (int i=0; i < 5; i++)
	{	recent_number.setNum(i+1);
		str = config->readEntry(recent_number.data(), "");
		if (!str.isEmpty() && QFile::exists(str.data()))
   	 	recentStringFileList.append(str.data());
	}
	
}

// ===========================================================================================================================
void KFileReplaceApp::saveProperties(KConfig* )
{
	if( doc->getTitle() != i18n("Untitled") && !doc->isModified()){
		return;
	}
	else{
		QString filename=doc->getPathName()+doc->getTitle();	
    config->writeEntry("filename",filename);
    config->writeEntry("modified",doc->isModified());
		
		const char* tempname = kapp->tempSaveName(filename);
		doc->saveDocument(tempname);
	}
}


// ===========================================================================================================================
void KFileReplaceApp::readProperties(KConfig*)
{
	QString filename = config->readEntry("filename","");
	bool modified = config->readBoolEntry("modified",false);
  if( modified ){
  	bool b_canRecover;
		QString tempname = kapp->checkRecoverFile(filename,b_canRecover);
  	
  	if(b_canRecover){
   			doc->openDocument(tempname);
   			doc->setModified();
   			QFileInfo info(filename);
   			doc->pathName(info.absFilePath());
   			doc->title(info.fileName());
   			QFile::remove(tempname);
		}
	}
 	else if(!filename.isEmpty()){
		doc->openDocument(filename);
	}
  QString caption=kapp->getCaption();	
  setCaption(caption+": "+doc->getTitle());
}		

// ===========================================================================================================================
bool KFileReplaceApp::queryClose()
{
	saveOptions();
	return true;
}

// ===========================================================================================================================
bool KFileReplaceApp::queryExit()
{
	saveOptions();
	return true;
}

/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATION
/////////////////////////////////////////////////////////////////////

// ===========================================================================================================================
void KFileReplaceApp::slotFileNewWindow()
{
  slotStatusMsg(i18n("Opening a new Application window..."));
	
	KFileReplaceApp* new_window= new KFileReplaceApp();
	new_window->show();
	
  slotStatusMsg(i18n("Ready."));
}

// ===========================================================================================================================
void KFileReplaceApp::slotFileNew()
{
	int nRes;
  slotStatusMsg(i18n("Creating new document..."));
	
	nRes = doc->newDocument();
	if (nRes == false)
		return;
		
  QString caption=kapp->getCaption();	
  setCaption(caption+": "+doc->getTitle());

	// Empty lists views
	view -> getStringsList() -> clear();
	view -> getResultList() -> clear();

  slotStatusMsg(i18n("Ready."));
}

// ===========================================================================================================================
void KFileReplaceApp::closeEvent (QCloseEvent *event)
{
	saveOptions();

	// Check Thread is not running
	if (g_bThreadRunning)
	{	KMsgBox::message(this, "KFileReplace", i18n("You can't quit before having finished the replacing operation."), KMsgBox::EXCLAMATION, i18n("Cancel"));
		return;
	}
	
	KTMainWindow::closeEvent (event);
}

// ===========================================================================================================================
void KFileReplaceApp::slotFileQuit()
{
	// Check Thread is not running
	if (g_bThreadRunning)
	{	KMsgBox::message(this, "KFileReplace", i18n("You can't quit before having finished the replacing operation."), KMsgBox::EXCLAMATION, i18n("Cancel"));
		return;
	}

  ///////////////////////////////////////////////////////////////////
  // exits the Application
	saveOptions();
	// close the first window, the list makes the next one the first again.
	// This ensures that queryClose() is called on each window to ask for closing
	KTMainWindow* w;
	if(memberList){
		for(w=memberList->first(); w; w=memberList->first()){
			// only close the window if the closeEvent is accepted. If the user presses Cancel on the saveModified() dialog,
			// the window and the application stay open.
			if(!w->close())
				break;
		}
	}	
}

// ===========================================================================================================================
void KFileReplaceApp::slotHelpWebSite()
{	if(vfork() > 0)
	{ // drop setuid, setgid
    setgid(getgid());
    setuid(getuid());

    execlp("kfmclient", "kfmclient", "exec", QString("http://perso.club-internet.fr/dupoux/sysexp/linux/kfilerep/").data(), 0);
    _exit(0);
  }
}

// ===========================================================================================================================
void KFileReplaceApp::slotHelpAbout()
{
  QMessageBox aboutmsg(this, "About KFileReplace");
  aboutmsg.setCaption(i18n("About KFileReplace..."));
  aboutmsg.setButtonText(1, i18n("OK"));
  aboutmsg.setText("\t           KFileReplace version "VERSION" \n\n\n"
                   "Main author: Franois Dupoux <fdupoux@lemel.fr>\n\n"
									 "German translator: Heiko Goller <heiko.goller@tuebingen.mpg.de>\n\n\n"
                   "http://perso.club-internet.fr/dupoux/sysexp/linux/kfilerep/\n");

  QPixmap pm;
  pm.load(KApplication::kde_datadir() + "/kfilereplace/pics/about.bmp");
  aboutmsg.setIconPixmap(pm);
  aboutmsg.show();

}


// ===========================================================================================================================
void KFileReplaceApp::slotViewToolBar()
{
  ///////////////////////////////////////////////////////////////////
  // turn Toolbar on or off
	if( view_menu->isItemChecked(ID_VIEW_TOOLBAR))
	  view_menu->setItemChecked(ID_VIEW_TOOLBAR, false);
	else
		view_menu->setItemChecked(ID_VIEW_TOOLBAR, true);
		
  enableToolBar();

  slotStatusMsg(i18n("Ready."));
}

// ===========================================================================================================================
void KFileReplaceApp::slotViewStatusBar()
{
  ///////////////////////////////////////////////////////////////////
  //turn Statusbar on or off
	if( view_menu->isItemChecked(ID_VIEW_STATUSBAR))
	  view_menu->setItemChecked(ID_VIEW_STATUSBAR, false);
	else
		view_menu->setItemChecked(ID_VIEW_STATUSBAR, true);

  enableStatusBar();

  slotStatusMsg(i18n("Ready."));
}


// ===========================================================================================================================
void KFileReplaceApp::slotStatusMsg(const char *text)
{
  ///////////////////////////////////////////////////////////////////
  // change status message permanently
  statusBar()->clear();
  statusBar()->changeItem(text, ID_STATUS_MSG );
}


// ===========================================================================================================================
void KFileReplaceApp::slotStatusHelpMsg(const char *text)
{
  ///////////////////////////////////////////////////////////////////
  // change status message of whole statusbar temporary (text, msec)
  statusBar()->message(text, 2000);
}

// ===========================================================================================================================
void KFileReplaceApp::commandCallback(int id_)
{
  switch (id_)
	{
    ON_CMD(ID_FILE_NEW_WINDOW,          slotFileNewWindow())
    ON_CMD(ID_FILE_NEW,                 slotFileNew())
    ON_CMD(ID_FILE_SEARCH,              slotFileSearch())
    ON_CMD(ID_FILE_REPLACE,             slotFileReplace())
    ON_CMD(ID_FILE_STOP,                slotFileStop())
    ON_CMD(ID_FILE_SAVE,                slotFileSave())
    ON_CMD(ID_FILE_QUIT,                slotFileQuit())

    ON_CMD(ID_STRINGS_ADD,              slotStringsAdd())
    ON_CMD(ID_STRINGS_DEL,              slotStringsDel())
		ON_CMD(ID_STRINGS_EMPTY,            slotStringsEmpty())
		ON_CMD(ID_STRINGS_EDIT,            	slotStringsEdit())
		ON_CMD(ID_STRINGS_SAVE,             slotStringsSave())
		ON_CMD(ID_STRINGS_LOAD,             slotStringsLoad())
		ON_CMD(ID_STRINGS_INVERTCUR,        slotStringsInvertCur())
		ON_CMD(ID_STRINGS_INVERTALL,        slotStringsInvertAll())

    ON_CMD(ID_RESULT_INFO,              slotResultInfo())
    ON_CMD(ID_RESULT_OPEN,              slotResultOpen())
		ON_CMD(ID_RESULT_DIROPEN,           slotResultDirOpen())
    ON_CMD(ID_RESULT_DELETE,            slotResultDelete())

    ON_CMD(ID_OPTIONS_RECURSIVE,        slotOptionsRecursive())
    ON_CMD(ID_OPTIONS_BAKUP,            slotOptionsBakup())
		ON_CMD(ID_OPTIONS_CASESENSITIVE,    slotOptionsCaseSensitive())

    ON_CMD(ID_VIEW_TOOLBAR,             slotViewToolBar())
    ON_CMD(ID_VIEW_STATUSBAR,           slotViewStatusBar())

    ON_CMD(ID_HELP_WEBSITE,             slotHelpWebSite())
    ON_CMD(ID_HELP_ABOUT,               slotHelpAbout())
  }

	// Activate or desactivate some commands if need
	updateCommands();

}


// ===========================================================================================================================
void KFileReplaceApp::statusCallback(int id_)
{
  switch (id_)
	{
    ON_STATUS_MSG(ID_FILE_NEW_WINDOW,				i18n("Opens a new application window"))
    ON_STATUS_MSG(ID_FILE_NEW,      				i18n("Creates a new document"))
    ON_STATUS_MSG(ID_FILE_SEARCH,						i18n("Runs search operation"))
    ON_STATUS_MSG(ID_FILE_REPLACE,					i18n("Runs replace operation"))
    ON_STATUS_MSG(ID_FILE_STOP,							i18n("Stop current operation"))
    ON_STATUS_MSG(ID_FILE_SAVE,							i18n("Save list of results in a html file"))
    ON_STATUS_MSG(ID_FILE_QUIT,							i18n("Exits"))
    ON_STATUS_MSG(ID_STRINGS_ADD,						i18n("Add a string to search or to replace"))
    ON_STATUS_MSG(ID_STRINGS_DEL,						i18n("Delete a string to search or to replace"))
    ON_STATUS_MSG(ID_STRINGS_EMPTY,					i18n("Delete all strings of the list"))
    ON_STATUS_MSG(ID_STRINGS_EDIT,					i18n("Modify text of selected string"))
    ON_STATUS_MSG(ID_STRINGS_SAVE,					i18n("Save all the strings of the list into a file"))
    ON_STATUS_MSG(ID_STRINGS_LOAD,					i18n("Load strings from a file"))
    ON_STATUS_MSG(ID_STRINGS_INVERTCUR,			i18n("Invert searched and replaced text for the selected string"))
    ON_STATUS_MSG(ID_STRINGS_INVERTALL,			i18n("Invert searched and replaced text for all strings"))
    ON_STATUS_MSG(ID_OPTIONS_RECURSIVE,			i18n("Search/Replace in all sub directories of the main directory"))
    ON_STATUS_MSG(ID_OPTIONS_BAKUP,					i18n("Create a bakup of the file before replacing strings inside"))
    ON_STATUS_MSG(ID_OPTIONS_CASESENSITIVE,	i18n("Lower and Uppers letter are differents"))
    ON_STATUS_MSG(ID_RESULT_INFO,						i18n("Informations about selected file"))
    ON_STATUS_MSG(ID_RESULT_OPEN,						i18n("Open selected file"))
    ON_STATUS_MSG(ID_RESULT_DIROPEN,				i18n("Open parent directory"))
    ON_STATUS_MSG(ID_RESULT_DELETE,					i18n("Delete selected file"))
    ON_STATUS_MSG(ID_HELP_WEBSITE,					i18n("Surf on the web page of KFileReplace"))
    ON_STATUS_MSG(ID_HELP_ABOUT,						i18n("About KFileReplace..."))
    ON_STATUS_MSG(ID_VIEW_TOOLBAR,					i18n("Enables / disables the actual Toolbar"))
    ON_STATUS_MSG(ID_VIEW_STATUSBAR,				i18n("Enables / disables the actual Statusbar"))
  }
}

// ===========================================================================================================================
void KFileReplaceApp::AddRecentStringFileToList(QString strFilename)
{
  int i;

	// Check file in not already in the list
	for (i=0; i < (int) recentStringFileList.count(); i++)
	{	if (strFilename == recentStringFileList.at(i)) // if file is already in the list
			return ;
  }

  if( recentStringFileList.count() >= 5)
    recentStringFileList.remove(4);
	
	recentStringFileList.insert(0, strFilename.data());

	// Redraw menu
  recentStringPopup -> clear();
  for (i = 0; i < (int) recentStringFileList.count(); i++)
    recentStringPopup -> insertItem(recentStringFileList.at(i));
}

// ===========================================================================================================================
/** Create the text with a size in KB, MB, or Bytes*/
QString formatSize(uint nSize)
{	QString strSize;
	double dSize;
	
	if (nSize < 1024) // In Bytes
	{	strSize.sprintf(i18n("%ld bytes"), nSize);
	}
  else if (nSize < 1024 * 1024) // In KiloBytes
	{	dSize = ((double) nSize) / ((double) 1024.0);
		strSize.sprintf(i18n("%.2f KB"), (float) dSize);
	}
  else if (nSize < 1024 * 1024 * 1024) // In MegaBytes
	{	dSize = ((double) nSize) / ((double) (1024.0 * 1024.0));
		strSize.sprintf(i18n("%.2f MB"), (float) dSize);
	}
  else // In GigaBytes
	{	dSize = ((double) nSize) / ((double) (1024.0 * 1024.0 * 1024.0));
		strSize.sprintf(i18n("%.2f GB"), (float) dSize);
	}

	return strSize;
}

























