/***************************************************************************
                          kfilecoderdoc.cpp  -  description                              
                             -------------------                                         
    begin                : ven avr 23 18:58:17 CEST 1999
                                           
    copyright            : (C) 1999 by Franois Dupoux                         
    email                : fdupoux@voila.fr                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#include <qmessagebox.h>
#include <qprogressdialog.h>
#include <qcstring.h>
#include <qfileinfo.h>

#include <kmessagebox.h>
#include <kpassdlg.h>
#include <kurl.h>
#include <kio/netaccess.h>

#include "kfilecoder.h"
#include "kfilecoderdoc.h"
#include "filegest.h"
                                           
#include <sys/vfs.h>


// ===========================================================================================================================
KFileCoderDoc::KFileCoderDoc(QObject *parent, const char *)
{
	m_bArchiveOpened = FALSE;
	*m_szArchivePath = 0;
	*m_szPassword = 0;
	m_nAlgorithmUsed = 0; // No algorith
	m_nFlags = 0;	
	m_mainWnd = (QWidget *) parent;
}

// ===========================================================================================================================
KFileCoderDoc::~KFileCoderDoc()
{

}

// ===========================================================================================================================
QFile *KFileCoderDoc::getArchiveFile()
{	return &m_fArchive;
}

// ===========================================================================================================================
int KFileCoderDoc::getNbFiles()
{	return m_nNbFiles;
}

// ===========================================================================================================================
int KFileCoderDoc::getAlgoUsed()
{	return m_nAlgorithmUsed;
}

// ===========================================================================================================================
void KFileCoderDoc::setListView (QListView *lv)
{	m_List = lv;	
}

// ===========================================================================================================================
int KFileCoderDoc::createArchive(const char *szFilename, const char *szPassword, int nAlgorithmUsed) // Create a new archive and write header: put m_szPassword, m_szArchivePAth, m_fdArchive
{
	QString strMess;
	int nRes;
	bool bRes;
	
	// Archive is already opened
	if (m_bArchiveOpened == TRUE)
	{	 KMessageBox::error( m_mainWnd, i18n("An archive is already opened. You must close it before."), "KFileCoder");
		return -1;
	}

 	// Initialize QFile m_fArchive
	m_fArchive.setName(szFilename);

 	// The file already exists: overwrite ?
	if (m_fArchive.exists())
	{	strMess.sprintf(i18n("The archive %s already exists. Do you want to overwrite it ? All data will be erased."), szFilename);
		if (KMessageBox::questionYesNo(m_mainWnd, strMess, "KFileCoder") == KMessageBox::No) // No clicked
			return -1;
	}

	// Creating the file
        bRes = m_fArchive.open(IO_ReadWrite | IO_Truncate);
	if (bRes == FALSE)
	{	strMess.sprintf(i18n("Unable to open the file %s for Reading and Writing."), szFilename);	
		KMessageBox::error( m_mainWnd, strMess, "KFileCoder");
		return -1;
	}

	// Put flags and datas
	m_bArchiveOpened = TRUE;
	strcpy(m_szArchivePath, szFilename);
	strcpy(m_szPassword, szPassword); // Must be done before writeHeader() !

	// Set default flags for this version of KFileCoder:
	m_nFlags = FLAG_CRCCHECKPASSWORD;

	// Write the header
	m_nAlgorithmUsed = nAlgorithmUsed;
	nRes = writeHeader();
	if (nRes == -1)
	{	closeArchive();
                return -1; // error
	}

	// Redraw the List View
	listView_Fill();

	// Set caption of the window
	QString strCaption;
	strCaption.sprintf("KFileCoder %s: %s", VERSION, m_szArchivePath);
	m_mainWnd -> setCaption(strCaption);

	return 0; // Success	
}


// ===========================================================================================================================
int KFileCoderDoc::openArchive(const char *szFilename, const char *szPassword /*=0*/) // Open the archive and read header: put m_szPassword, m_szArchivePAth, m_fdArchive
{	
	QString strMess;
	int nRes;
	bool bRes;
	QCString strPass;

	// Archive is already opened
	if (m_bArchiveOpened == TRUE)
	{	KMessageBox::error(m_mainWnd, i18n("An archive is already opened. You must close it before."), "KFileCoder");
		return -1;
	}

 	// Initialize QFile m_fArchive
	m_fArchive.setName(szFilename);

	// The file doesn't exists: create ?
	if (!m_fArchive.exists())
	{	strMess.sprintf(i18n("The archive %s doesn't exists. Do you want to create it ? ."), szFilename);
		if (KMessageBox::questionYesNo(m_mainWnd, strMess, "KFileCoder") == KMessageBox::No) // No clicked
			return -1;

                // Yes ==> Create a new archive

                // Dialog to ask password ==> szPassword
                if (szPassword) // If we know the password
                {       createArchive(szFilename, szPassword, ALGO_ALG1XOR);
                }
                else // don't know the password
                {       nRes = KPasswordDialog::getNewPassword(strPass, i18n("Please, enter the password used to encode files."));
	                if (nRes != KPasswordDialog::Accepted)
	                        return -1;

                        createArchive(szFilename, strPass.data(), ALGO_ALG1XOR);
                }
		return 0;
	}

	// Opening the file
        bRes = m_fArchive.open(IO_ReadWrite);
	if (bRes == FALSE)
	{	strMess.sprintf(i18n("Unable to open the file %s for Reading and Writing."), szFilename);	
		KMessageBox::error( m_mainWnd, strMess, "KFileCoder");
		return -1;
	}

	// Put flags and datas
	m_bArchiveOpened = TRUE;
	strcpy(m_szArchivePath, szFilename);

        if (szPassword) // If we know the password
	{       strcpy(m_szPassword, szPassword); // Must be done before ReadHeader()!
        }
        else // Don't know the password
	{       // Dialog to ask password ==> szPassword
	        nRes = KPasswordDialog::getNewPassword(strPass, i18n("Please, enter the password used to encode files."));
	        if (nRes != KPasswordDialog::Accepted)
	                return -1;
                strcpy(m_szPassword, strPass.data()); // Must be done before ReadHeader()!
        }

	// Read the header
	nRes = readHeader();
	if (nRes == -1)
	{	closeArchive();
                return -1; // error
	}

	// Redraw the List View
	listView_Fill();

	// Set caption of the window
	QString strCaption;
	strCaption.sprintf("KFileCoder %s: %s", VERSION, m_szArchivePath);
	m_mainWnd -> setCaption(strCaption);

	return 0; // Success	
}

// ===========================================================================================================================
int KFileCoderDoc::closeArchive(bool bClearList /*=true*/) // Close the current Archive
{
	// No rchive opened
	if (m_bArchiveOpened == FALSE)
	{ return -1;
	}

	m_bArchiveOpened = FALSE;
	m_fArchive.close();
	m_nNbFiles = 0;
	*m_szArchivePath = 0;
	*m_szPassword = 0;
	m_nFlags = 0;	

	if (bClearList)
		m_List -> clear();

	// Update menu & toolbar commands
	((KFileCoderApp *) m_mainWnd) -> updateCommandsAccess();

	// Set caption of the window
	QString strCaption;
	strCaption.sprintf("KFileCoder %s", VERSION);
	m_mainWnd -> setCaption(strCaption);

	return 0;
}

// ===========================================================================================================================
int KFileCoderDoc::writeHeader() // When creating new archive
{
	ArchiveHeader head;
	int nRes;

	// Archive is not opened
	if (m_bArchiveOpened != TRUE)
	{	KMessageBox::error(m_mainWnd, i18n("An archive must be opened to write header."), "KFileCoder");
		return -1;
	}

	// Go to the begin of the file
	m_fArchive.at(0);

	// Prepare datas of header which will be written to the archive
	sprintf (head.szPgm, "KFileCoder");
	head.nAlgorithmUsed = m_nAlgorithmUsed;

	// Set flags
	head.nFlags = m_nFlags;

	// Calculate the CRC for the password test
	if (head.nFlags & FLAG_CRCCHECKPASSWORD) // TRUE if the CRC which allow to check the password is written in the archive header	
	{	head.nCrcForPasswordTest = getPasswordTestCrc();
	}

	// Write the header
	nRes = m_fArchive.writeBlock ((const char*) &head, sizeof(ArchiveHeader));
	m_fArchive.flush();
	if (nRes == -1)
	{	KMessageBox::error( m_mainWnd, i18n("Can't write header of Archive."), "KFileCoder");
		return -1;
	}

	return 0;
}

// ===========================================================================================================================
int KFileCoderDoc::readHeader()	// When opening an archive		
{	
	ArchiveHeader head;
	uint nCrcForPasswordTest;
	int nRes;

	// Archive is net opened
	if (m_bArchiveOpened != TRUE)
	{	KMessageBox::error( m_mainWnd, i18n("An archive must be opened to read header."), "KFileCoder");
		return -1;
	}

	// Go to the begin of the file
	m_fArchive.at(0);

	nRes = m_fArchive.readBlock ((char*) &head, sizeof(ArchiveHeader));
	if (nRes == -1)
	{	KMessageBox::error( m_mainWnd, i18n("Can't read header of Archive."), "KFileCoder");
		return -1;
	}

	// Compare header with "KFileCoder document"
	head.szPgm[11] = 0;
	if (strcmp(head.szPgm, "KFileCoder") != 0) // The opened archive is not a KFileCoder archive
	{	KMessageBox::error(m_mainWnd, i18n("This file is not a KFileCoder archive."), "KFileCoder");
		return -1;
	}	

	// Get flags
	m_nFlags = head.nFlags;

	// if flag is true, Calculate the CRC for the password test
	if (head.nFlags & FLAG_CRCCHECKPASSWORD) // TRUE if the CRC which allow to check the password is written in the archive header
	{	nCrcForPasswordTest = getPasswordTestCrc();
		if (head.nCrcForPasswordTest != nCrcForPasswordTest) // Compare original password, and current password
		{	KMessageBox::error(m_mainWnd, i18n("Incorrect password."), "KFileCoder");
			return -1;
		}	
	}	
	
	// Read the number of the algorithm used to encode files
	m_nAlgorithmUsed = head.nAlgorithmUsed;
	if (head.nAlgorithmUsed > ALGO_LASTSUPPORTED)
	{	KMessageBox::error (m_mainWnd, i18n("The algorith which was used to encode the archive is not supported in this version of KFileCoder. Please, download a new one on: http://perso.club-internet.fr/dupoux/sysexp/linux/kfileocd/."), "KFileCoder");
		return -1;
	}

	return 0;
}


// ===========================================================================================================================
int KFileCoderDoc::addFile(KURL urlFile)
{	int nRes;
	bool bRes;	
	QString strMess;
        QString strFilename;
	QFile fFileToAdd;
        char szFilename[MAXPATHLEN];
	char szFileName[MAXPATHLEN];
	char szFilePath[MAXPATHLEN];
	uint nOriginalCrc = 0;
	uint nFileSize;
	uint nDiskFreeSpace;
        QFileInfo fi;

        // Download file if need (if url is "http://...")
	if (!(KIO::NetAccess::download(urlFile, strFilename))) // Open the Archive
                return -1;

	// Check it's not a directory
	fi.setFile(strFilename);
	if (fi.isDir())
	{	KMessageBox::error(m_mainWnd, i18n("Can't open directories."));
		return -1;
	}

	// Check archive an is opened
	if (m_bArchiveOpened != TRUE)
	{	KMessageBox::error (m_mainWnd, i18n("An archive must be opened to add a file."), "KFileCoder");
		return -1;
	}

	// Calculate FileName and FilePath from szFilename
        sprintf(szFilename, "%s", strFilename.data());
        getFileName(szFileName, szFilename);
	getFilePath(szFilePath, szFilename);

	// Check the file is not already in the archive
	bRes = doesFileExistsInArchive(szFileName, szFilePath);
	if (bRes == true)
	{	strMess.sprintf(i18n("The file %s/%s is already present in the archive. KFileCoder can't add two same files."), szFilePath, szFileName);
		KMessageBox::error (m_mainWnd, strMess, "KFileCoder");
		return -1;
	}

	// --------- Open file to add ------------------------
	fFileToAdd.setName(szFilename);
	bRes = fFileToAdd.open(IO_ReadOnly | IO_Raw);

	if (bRes == FALSE)
        {	strMess.sprintf(i18n("Can't open file %s for reading."), szFilename);
		KMessageBox::error (m_mainWnd, strMess, "KFileCoder");
		return -1;
	}

	// --------- Write header of file to add ------------------------	
	nFileSize = fFileToAdd.size();

	// Check there is enought free disk space
	nRes = getDiskFreeSpaceForFile(&nDiskFreeSpace, m_szArchivePath);
	if (nRes != -1 && nDiskFreeSpace  < nFileSize)
	{	KMessageBox::error (m_mainWnd, i18n("There is not enought disk free space, this operation need."), "KFileCoder");
   	        return -1;
	}

	// Write "Header" for the file
	nRes = writeFileString(&m_fArchive, szFileName, true, m_szPassword);
	if (nRes == -1)
		return -1;

	nRes = writeFileString(&m_fArchive, szFilePath, true, m_szPassword);
	if (nRes == -1)
		return -1;

	nRes = m_fArchive.writeBlock ((char*) &nFileSize, sizeof(uint));	
	if (nRes == -1)	
        {	strMess.sprintf(i18n("Can't write file %s to archive."), szFilename);
		KMessageBox::error (m_mainWnd, strMess, "KFileCoder");
		fFileToAdd.close();
		return -1;
	}

	// Go to the end of the archive
	m_fArchive.atEnd();

	// =+=+=+=+=+=+=+=+=+=+ Add encoding algorithm function =+=+=+=+=+=+=+=+=+=+
	nRes = 0;
	switch(m_nAlgorithmUsed)
	{	case ALGO_ALG1XOR:
			nRes = addFileAlgo1(szFilename, &fFileToAdd, &nOriginalCrc);
			break;

		default:
			break;
	}
	// =+=+=+=+=+=+=+=+=+=+ Add encoding algorithm function =+=+=+=+=+=+=+=+=+=+

	if (nRes == -1)	
	{	fFileToAdd.close();
		return -1;
	}

	// --------- Write CRC of the file -------------------
	m_fArchive.writeBlock ((char*) &nOriginalCrc, sizeof(uint));
	if (nRes == -1)	
        {	strMess.sprintf(i18n("Can't write file %s to archive."), szFilename);
		KMessageBox::error(m_mainWnd, strMess, "KFileCoder");
		fFileToAdd.close();
		return -1;
	}

	// Close file added to archive
 	fFileToAdd.close();
	
	// Add file to the ListView
	listView_AddItem(szFileName, nFileSize, nOriginalCrc, szFilePath);

	m_nNbFiles++;

	return 0;
}

// ===========================================================================================================================
int KFileCoderDoc::addFileAlgo1(const char *szFilename, QFile *fFileToAdd, uint *nOriginalCrc)
{
	int nPassLen = strlen(m_szPassword);
	int passPos = 0; // Position in password
	QString strMess;
	int nRes;
	int i;
	const int nBufSize = 512;
	char cBuffer[nBufSize];
	int nBytesRead;
	
	// Initializations
	*nOriginalCrc = 0;
	passPos = 0;

	do
	{	
  	// Reads a block from file to add
		nBytesRead = fFileToAdd -> readBlock(cBuffer, nBufSize);
		if (nBytesRead == -1)
  	{	strMess.sprintf(i18n("Can't read file %s."), szFilename);
			KMessageBox::error (m_mainWnd, strMess, "KFileCoder");
			return -1;
		}

		// Encode block
		for (i=0; i < nBytesRead; i++)
		{
			// Calculate CRC of file added	
			(*nOriginalCrc) += cBuffer[i];

			// Encode character with XOR algorithm
			cBuffer[i] ^= m_szPassword[passPos];

			passPos++;
			if (passPos == nPassLen) // if end of password, goto begin
					passPos = 0;
		}
			
 		nRes = m_fArchive.writeBlock(cBuffer, nBytesRead);
		
		if (nRes != nBytesRead)
  	{	strMess.sprintf(i18n("Can't write file %s to archive."), szFilename);
			KMessageBox::error (m_mainWnd, strMess, "KFileCoder");
			return -1;
		}

	} while (nBytesRead == nBufSize);

	return 0;
}

// ===========================================================================================================================
int KFileCoderDoc::extractFiles(const char *szDestPath, bool bOnlySelFiles)
{	char szFileName[MAXPATHLEN];
	char szFilePath[MAXPATHLEN];
	char szFullpath[MAXPATHLEN];
	char szPath[MAXPATHLEN];
	uint nFileSize;
	uint nFileCrc;
	QString strMess;
	QFile fFileToDecode;
	uint nExtractedCrc = 0;
	bool bRes;
	int nRes;
	bool bOverwriteYesToAll = false;
	bool bOverwriteNoToAll = false;
	uint nDiskFreeSpace;

	// Check archive an is opened
	if (m_bArchiveOpened != TRUE)
	{	KMessageBox::error(m_mainWnd, i18n("An archive must be opened to extract all its files."), "KFileCoder");
		return -1;
	}

	// Go to the begin of the archive, after Header
  m_fArchive.at(sizeof(ArchiveHeader));
 	
	// Draw the progress dialog
	QProgressDialog dlgProgress(i18n("Extracting files of archive.."),0, m_fArchive.size(), m_mainWnd,"",true);
  dlgProgress.setCaption("KFileCoder");
	dlgProgress.show();
	dlgProgress.setProgress(m_fArchive.at());

	while (m_fArchive.at() < m_fArchive.size()) // while !feof
	{	
   	// ----- Read "Header" of the file -----
		nRes = readFileString(&m_fArchive, szFileName, true, m_szPassword);
		if (nRes == -1)
			return -1;

		nRes = readFileString(&m_fArchive, szFilePath, true, m_szPassword);
		if (nRes == -1)
			return -1;

		nRes = m_fArchive.readBlock ((char*) &nFileSize, sizeof(uint));	
		if (nRes == -1)	
  	{	KMessageBox::error (m_mainWnd, i18n("Can't read archive."), "KFileCoder");
			return -1;
		}

		// Calculate szFullpath (path and name)
		if (szFilePath[0] == '/')
			sprintf (szFullpath, "%s%s/%s", szDestPath, szFilePath, szFileName);
		else
			sprintf (szFullpath, "%s/%s/%s", szDestPath, szFilePath, szFileName);

		// Calculate szPath (path only) to create directory
		if (szFilePath[0] == '/')
			sprintf (szPath, "%s%s", szDestPath, szFilePath);
		else
			sprintf (szPath, "%s/%s", szDestPath, szFilePath);

  	// Check there is enought free disk space
		nRes = getDiskFreeSpaceForFile(&nDiskFreeSpace, szFullpath);
		if (nRes != -1 && nDiskFreeSpace  < nFileSize)
		{	KMessageBox::error(m_mainWnd, i18n("There is not enought disk free space, this operation need."), "KFileCoder");
   		return -1;
		}

		// Make all dirs of path: create directory if they doesn'y exists
		createAllDirsOfPath(szPath);
	    	
		// Open the file to create decoded
		fFileToDecode.setName(szFullpath);

		// -------------- The file already exists: overwrite ? --------------------
		if (fFileToDecode.exists())
		{	nRes = 0;	

      // Ask question "Owerwrite ?" only if YesToAll or NoToAll have not been selected before
			if (bOverwriteYesToAll == false && bOverwriteNoToAll == false)
			{	strMess.sprintf(i18n("The file %s already exists. Do you want to overwrite it ? All data will be erased."), szFullpath);
				
				nRes = KMessageBox::questionYesNo(m_mainWnd, strMess, "KFileCoder"); // Do not support "Yes/No for all files"
			
				/* KDE 1.x version
				KMsgBox msg(m_mainWnd, "KFileCoder", strMess, KMessageBox::QUESTION, i18n("Yes"), i18n("No"), i18n("Yes to all"), i18n("No to all"));
	    			nRes = dlg.exec();
				if (nRes == 3) // Yes to all
					bOverwriteYesToAll = true;
				if (nRes == 4) // No to all
					bOverwriteNoToAll = true;*/
			}

			if (bOverwriteNoToAll == true || nRes == KMessageBox::No) // We must not owerwrite ==> Skip file in archive
			{	m_fArchive.at(m_fArchive.at() + nFileSize); // skip file contents
				m_fArchive.at(m_fArchive.at() + sizeof(uint)); // skip file CRC
				continue; // return to the begin of the while()
			}
		}


		// =+=+=+=+=+=+=+=+=+=+ Extracting and decoding with algorithm function =+=+=+=+=+=+=+=+=+=+
		if (bOnlySelFiles == false || isFileSelected(szFileName, szFilePath) == true) // If we must extract file
		{ // Creating the file
  			bRes = fFileToDecode.open(IO_WriteOnly | IO_Truncate);
			if (bRes ==  FALSE)
			{	strMess.sprintf(i18n("Unable to open the file %s for Writing."), szFullpath);	
				KMessageBox::error (m_mainWnd, strMess, "KFileCoder");
				return -1;
			}
			nRes = 0;

			// Extract All Files: calling the function which Extract all Files witch the current Algorithm
			switch(m_nAlgorithmUsed)
			{	case ALGO_ALG1XOR:
					nRes = extractFilesAlgo1(szFullpath, &fFileToDecode, &nExtractedCrc, nFileSize);
					break;

				default:
					break;
			}

			if (nRes == -1)
			{	fFileToDecode.close();
				return -1;
			}

			// =+=+=+=+=+=+=+=+=+=+ Extracting and decoding with algorithm function =+=+=+=+=+=+=+=+=+=+

			// Close File to Decode
			fFileToDecode.close();	

			// ----- Read File CRC --------------
			nRes = m_fArchive.readBlock ((char*) &nFileCrc, sizeof(uint));	
			if (nRes == -1)	
  		{	KMessageBox::error (m_mainWnd, i18n("Can't read archive."), "KFileCoder");
				fFileToDecode.close();
				return -1;
			}

			// Compare real CRC and good CRC
  		if (nExtractedCrc != nFileCrc) // CRC are differents
  		{	strMess.sprintf(i18n("The file %s was extracted, but its CRC is not the same as original. There was a error in datas, or password is bad."), szFullpath);
				KMessageBox::error(m_mainWnd, strMess, "KFileCoder");
			}

		}
		else // If we mustn't extract file ==> Skip it
		{	m_fArchive.at(m_fArchive.at() + nFileSize); // skip file data
			m_fArchive.at(m_fArchive.at() + sizeof(uint)); // skip file CRC
		}

		// Progress Dialog
		dlgProgress.setProgress(m_fArchive.at());
	}

	return 0;
}

// ===========================================================================================================================
int KFileCoderDoc::extractFilesAlgo1(const char *szFullpath, QFile *fFileToDecode, unsigned int *nExtractedCrc, uint nFileSize)
{
	const int nBufSize = 512;
	QString strMess;
	char cBuffer[nBufSize];
	int nPassLen = strlen(m_szPassword);
	int passPos = 0; // Position in password
	int nBytesToRead;
	int nBytesRead;
	int nRes;

	passPos = 0;
	*nExtractedCrc = 0; 	
	uint i=0;
	int j=0;

	while(i < nFileSize)
	{	
    if ( (nFileSize - i) >= nBufSize) // If data to read in the file we are extracting < BufSize
			nBytesToRead = nBufSize;
		else
			nBytesToRead = nFileSize - i;
			
  	// Reads a character from archive
		nBytesRead = m_fArchive.readBlock(cBuffer, nBytesToRead);
		if (nBytesRead == -1)
  	{	KMessageBox::error (m_mainWnd, i18n("Can't read archive."), "KFileCoder");
			return -1;
		}
		i += nBytesRead;

		// Decode block
		for (j=0; j < nBytesRead; j++)
		{	// Decode character with XOR algorithm
			cBuffer[j] ^= m_szPassword[passPos];
				
			// Calculate CRC of file added	
			*nExtractedCrc += cBuffer[j];

			passPos++;
			if (passPos == nPassLen) // if end of password, goto begin
				passPos = 0;
		}
			
		nRes = fFileToDecode -> writeBlock(cBuffer, nBytesRead);		

		if (nRes == -1)
  	{	strMess.sprintf(i18n("Can't write file %s."), szFullpath);
			KMessageBox::error (m_mainWnd, strMess, "KFileCoder");
			return -1;
		}
	}	

	return 0;
}

// ===========================================================================================================================
int KFileCoderDoc::deleteSelectedFilesInArchive()			
{
	// Copy the archive to a new one: copy only files which are not selected, using function isFileSelected();
	// Then move the current archive, and recreate a new one, from the old which is moved, by copying selected files

	char szOldArchive[MAXPATHLEN];
	char szNewArchive[MAXPATHLEN];
	char szPassword[MAX_PASSLEN];
	char szFileName[MAXPATHLEN];
	char szFilePath[MAXPATHLEN];
	const int nBufSize = 2048; // Must be more than sizeof(ArchiveHeader) and nFileHeadSize
	char cBuffer[nBufSize];
	int nFileHeadSize;
	int nBytesToRead;
	int nBytesRead;
	int nTotalBytesToCopy;
	uint nArchiveSize;
	uint nFileSize;
	int nRes;
	bool bRes;
	uint nDiskFreeSpace;
	QString strMess;

	// Check archive an is opened
	if (m_bArchiveOpened != TRUE)
	{	KMessageBox::error(m_mainWnd, i18n("An archive must be opened to delete files."), "KFileCoder");
		return -1;
	}

	// If no file selected, cancel
	if (getNbSelectedFiles() == 0)
	{	KMessageBox::error (m_mainWnd, i18n("There are no selected files."), "KFileCoder");
		return -1;
	}	

	// Initializations
	sprintf (szOldArchive, "%s", m_szArchivePath);
	sprintf (szNewArchive, "%s-new", m_szArchivePath); 	
	nArchiveSize = m_fArchive.size();
	sprintf (szPassword, "%s", m_szPassword);

  // Check there is enought free disk space
	nRes = getDiskFreeSpaceForFile(&nDiskFreeSpace, szNewArchive);
	if (nRes != -1 && nDiskFreeSpace < nArchiveSize)
	{	KMessageBox::error (m_mainWnd, i18n("There is not enought disk free space, this operation need."),"KFileCoder");
   	return -1;
	}

	// Close the archive
	closeArchive(false);

	// Open the files
	QFile fOld, fNew;
	fOld.setName(szOldArchive);
	fNew.setName(szNewArchive);

  bRes = fOld.open(IO_ReadOnly | IO_Raw);
	if (bRes == FALSE)
	{	KMessageBox::error (m_mainWnd, i18n("Can't delete files from archive."), "KFileCoder");
		return -1;
	}

  bRes = fNew.open(IO_ReadWrite | IO_Truncate);
	if (bRes == FALSE)
	{	KMessageBox::error (m_mainWnd, i18n("Can't delete files from archive."), "KFileCoder");
		return -1;
	}

 	// +=+=+=+=+=+=+=+= BEGIN: Copy the header and the files from OldArchive to NewArchive +=+=+=+=+=+=+=+=


	// 1. Copy the header
	fOld.at(0);
	fNew.at(0);

	nRes = fOld.readBlock (cBuffer, sizeof(ArchiveHeader));	
	if (nRes != sizeof(ArchiveHeader))	
  {	KMessageBox::error (m_mainWnd, i18n("Can't read old archive."), "KFileCoder");
		return -1;
	}

	nRes = fNew.writeBlock (cBuffer, sizeof(ArchiveHeader));	
	if (nRes != sizeof(ArchiveHeader))	
  {	KMessageBox::error( m_mainWnd, i18n("Can't write new archive."), "KFileCoder");
		fOld.close();
		fNew.close();
		return -1;
	}

	// 2. Copy the files if they are selected
	while (fOld.at() < fOld.size()) // while !feof
	{	
		nRes = readFileString(&fOld, szFileName, true, szPassword);
		if (nRes == -1)
			return -1;

		nRes = readFileString(&fOld, szFilePath, true, szPassword);
		if (nRes == -1)
			return -1;

		nRes = fOld.readBlock ((char*) &nFileSize, sizeof(uint));	
		if (nRes == -1)	
  	{	KMessageBox::error (m_mainWnd, i18n("Can't read old archive."), "KFileCoder");
			fOld.close();
			fNew.close();
			return -1;
		}

		if (isFileSelected(szFileName, szFilePath) == false) // If file is not selected ==> Do not delete ==> Copy it to new archive
		{	nFileHeadSize = sizeof(int) + strlen(szFileName) + sizeof(int) + strlen(szFilePath) + sizeof(uint);
			nTotalBytesToCopy = nFileHeadSize + nFileSize + sizeof(uint); // File Header(NameSize, Name, PathSize, Path, File Size) + File Contents + File CRC 	

			fOld.at ( fOld.at() - nFileHeadSize); // Go before file header

      // ******* Copy (nFileSize + sizeof(uint)) Bytes: file contents + crc *******
			while(nTotalBytesToCopy > 0) // Bytes not already copied
			{	
    		if (nTotalBytesToCopy >= nBufSize) // If data to read in the file we are extracting < BufSize
					nBytesToRead = nBufSize;
				else // nTotalBytesToCopy < nBufSize
					nBytesToRead = nTotalBytesToCopy;
			
  			// Reads a block from old archive
				nBytesRead = fOld.readBlock(cBuffer, nBytesToRead);
				
				if (nBytesRead != nBytesToRead)
  			{	KMessageBox::error (m_mainWnd, i18n("Can't read old archive."), "KFileCoder");
					fOld.close();
					fNew.close();
					return -1;
				}
	
				nTotalBytesToCopy -= nBytesRead;

				nRes = fNew.writeBlock(cBuffer, nBytesRead);		

				if (nRes != nBytesRead)
  			{	KMessageBox::error (m_mainWnd, i18n("Can't write new archive."), "KFileCoder");
					fOld.close();
					fNew.close();
					return -1;
				}
			}	
		}
		else // File not selected ==> Skip it
		{
  		fOld.at( fOld.at() + nFileSize); // Skip file contents
  		fOld.at( fOld.at() + sizeof(uint)); // Skip file crc
		}

	}

 	// +=+=+=+=+=+=+=+= END: Copy the header and the files from OldArchive to NewArchive +=+=+=+=+=+=+=+=

	// Close files
	fOld.close();
	fNew.close();

	// If operation succedded: Delete Old Archive
	fOld.remove();

	// If operation succedded: Rename the new archive to szOldArchive
	nRes = rename(szNewArchive, szOldArchive);
	if (nRes == -1)
	{	strMess.sprintf(i18n("Can't delete files from archive."), m_szArchivePath);	
		KMessageBox::error (m_mainWnd, strMess, "KFileCoder");
   	        return -1;
	}
	
        // Open the new archive
	openArchive(szOldArchive, szPassword);

	return 0;
}

// ===========================================================================================================================
int KFileCoderDoc::listView_Fill()
{
	char szFileName[MAXPATHLEN];
	char szFilePath[MAXPATHLEN];
	uint nFileSize;
	uint nFileCrc;
	int nRes;

	// Check archive an is opened
	if (m_bArchiveOpened != TRUE)
		return -1;
	
	// Empty the list
	m_List -> clear();
	m_nNbFiles = 0;

	// Go to the begin of the archive, after Header
  m_fArchive.at(sizeof(ArchiveHeader));

	while (m_fArchive.at() < m_fArchive.size()) // while !feof
	{	
   	// ----- Read "Header" of the file -----
		nRes = readFileString(&m_fArchive, szFileName, true, m_szPassword, false);
		if (nRes == -1)
			return -1;

		nRes = readFileString(&m_fArchive, szFilePath, true, m_szPassword, false);
		if (nRes == -1)
			return -1;

		nRes = m_fArchive.readBlock ((char*) &nFileSize, sizeof(uint));	
		if (nRes == -1)	
			return -1;

		// ----- Skip the file contents -----
  	m_fArchive.at( m_fArchive.at() + nFileSize);
		
		// ----- Read File CRC --------------
		nRes = m_fArchive.readBlock ((char*) &nFileCrc, sizeof(uint));	
		if (nRes == -1)	
			return -1;

		// Add Item to list View
		listView_AddItem(szFileName, nFileSize, nFileCrc, szFilePath);
		m_nNbFiles++;
	}
		
	return 0;
}

// ===========================================================================================================================
int KFileCoderDoc::listView_AddItem(const char *szName, uint nSize, uint nCrc, const char *szPath)
{
	char szCrc[128];

	// Prepare text to add
	sprintf (szCrc, "%.8X", nCrc);
	
	// Add item to list
	QListViewItem *lvi;

	lvi = new QListViewItem(m_List);
  	CHECK_PTR( lvi );

	lvi -> setText(0, szName);
	lvi -> setText(1, formatSize(nSize).data()); // Convert uint into a string in the good format (Bytes or KBytes or MBytes)
	lvi -> setText(2, szCrc);
	lvi -> setText(3, szPath);
	
	//lvi -> SetPixmap(0, *pmm); // To add icons in the ListView: icon of the type of documents

	// Update menu & toolbar commands
	((KFileCoderApp *) m_mainWnd) -> updateCommandsAccess();
	
	return 0;
}

// ===========================================================================================================================
uint KFileCoderDoc::getPasswordTestCrc() // Says if password is correct
{
	uint nCrc;
	const char *szDecodedData = "KFileCoder encodes files. It works under KDE (K Desktop Environment)."
												"It can code many files in an archive, and it encodes the data with a password.";
	char szEncodedData[512];
	int nDataLen;
	int nPassLen;
	int passPos;
	int i;

	passPos = 0;
	nCrc = 0;
	nDataLen = strlen(szDecodedData);
	nPassLen = strlen(m_szPassword);

	for (i=0; i < nDataLen; i++)
	{	szEncodedData[i] = szDecodedData[i] ^ m_szPassword[passPos];
		passPos++;
		if (passPos == nPassLen)
			passPos = 0;
		nCrc += szEncodedData[i];
	}

	return nCrc;
}

// ===========================================================================================================================
int KFileCoderDoc::encodeString(char *szDest, const char *szSource, int nStrLen, const char *szPassword) // Encode and decode a string using m_szPassword
{
	int passPos;
	int nPassLen;
	int i;
	
	passPos = 0;
	nPassLen = strlen(szPassword);

	for (i=0; i < nStrLen; i++)
	{	szDest[i] = szSource[i] ^ szPassword[passPos];
		passPos++;
		if (passPos == nPassLen)
			passPos = 0;	
	}

	szDest[nStrLen] = '\0'; // End of string
	return 0;
}

// ===========================================================================================================================
int KFileCoderDoc::writeFileString(QFile *qfOut, const char *szString, bool bEncode, const char *szPassword, bool bErrMsg /*=true*/)
{
	char szEncodedString[MAXPATHLEN];
	int nRes;
	int nLen;

	nLen = strlen(szString);

	if (bEncode)
		encodeString(szEncodedString, szString, nLen, szPassword);
	else
		strcpy(szEncodedString, szString);


	// Write size of the string
	nRes = qfOut -> writeBlock ((const char *)&nLen, sizeof(int));	
	if (nRes == -1)	
  {	if(bErrMsg)
	 		KMessageBox::error (m_mainWnd, i18n("Can't write to file."), "KFileCoder");
		return -1;
	}
	
	// Write the string
	nRes = qfOut -> writeBlock (szEncodedString, nLen);	
	if (nRes == -1)	
  {	if(bErrMsg)
	 		KMessageBox::error (m_mainWnd, i18n("Can't write to file."), "KFileCoder");
		return -1;
	}	

	return 0;
}

// ===========================================================================================================================
int KFileCoderDoc::readFileString(QFile *qfIn, char *szString, bool bEncode, const char *szPassword, bool bErrMsg /*=true*/)
{
	char szEncodedString[MAXPATHLEN];
	int nRes;
	int nLen;

	// Read size of the string
	nRes = qfIn -> readBlock ((char *)&nLen, sizeof(int));	
	if (nRes == -1)	
  {	if(bErrMsg)
	 		KMessageBox::error (m_mainWnd, i18n("Can't read from file."), "KFileCoder");
		return -1;
	}
	
	// Read the string
	nRes = qfIn -> readBlock (szEncodedString, nLen);	
	if (nRes == -1)	
  {	if(bErrMsg)
	 		KMessageBox::error (m_mainWnd, i18n("Can't read from file."), "KFileCoder");
		return -1;
	}	

	szEncodedString[nLen] = '\0'; // End of the string

	if (bEncode)
		encodeString(szString, szEncodedString, nLen, szPassword);
	else
		strcpy(szString, szEncodedString);

	szString[nLen] = '\0'; // End of the string

	return 0;
}

// ===========================================================================================================================
int KFileCoderDoc::getDiskFreeSpaceForFile(uint *nAvailDiskSpace, const char *szFilename)
{
	int nRes;
	struct statfs fsInfo;

	*nAvailDiskSpace = 0;

	nRes = statfs(szFilename, &fsInfo);
	if (nRes == -1)	
		return -1;	

  *nAvailDiskSpace = fsInfo.f_bavail * fsInfo.f_bsize;

	return 0;
}

// ===========================================================================================================================
bool KFileCoderDoc::isFileSelected(const char *szFileName, const char *szFilePath)
{
	QListViewItem *lviCurItem;
	QListViewItem *lviFirst;
	
	lviCurItem = lviFirst = m_List -> firstChild();
	if (lviCurItem == NULL)
	{	return false;
	}

	do
	{	if ( strcmp(szFileName, lviCurItem -> text(0)) == 0 && strcmp(szFilePath, lviCurItem -> text(3)) == 0 )
		{	return (lviCurItem -> isSelected());
		}

    lviCurItem = lviCurItem -> nextSibling();
	} while(lviCurItem && lviCurItem != lviFirst);

	return false; // Error: item not in the list
}

// ===========================================================================================================================
int KFileCoderDoc::getNbSelectedFiles()
{
	int nSel = 0;
	QListViewItem *lviCurItem;
	QListViewItem *lviFirst;
	
	lviCurItem = lviFirst = m_List -> firstChild();
	if (lviCurItem == NULL)
		return false;

	do
	{	if (lviCurItem -> isSelected())
			nSel++;

    lviCurItem = lviCurItem -> nextSibling();
	} while(lviCurItem && lviCurItem != lviFirst);

	return nSel; // Error: item not in the list
}

// ===========================================================================================================================
bool KFileCoderDoc::doesFileExistsInArchive(const char *szFileName, const char *szFilePath)
{
	QListViewItem *lviCurItem;
	QListViewItem *lviFirst;
	
	lviCurItem = lviFirst = m_List -> firstChild();
	if (lviCurItem == NULL)
		return false;

	do
	{	if ( strcmp(szFileName, lviCurItem -> text(0)) == 0 && strcmp(szFilePath, lviCurItem -> text(3)) == 0 )
		{	return true;
		}

    lviCurItem = lviCurItem -> nextSibling();
	} while(lviCurItem && lviCurItem != lviFirst);

	return false;
}
































































































































