/***************************************************************************
 *
 * knetworkmanager-devicestore_dbus.cpp - A NetworkManager frontend for KDE
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Timo Hoenig        <thoenig@suse.de>, <thoenig@nouse.net>
 *         Valentine Sinitsyn <e_val@inbox.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#include "knetworkmanager.h"
#include "knetworkmanager-dbus.h"
#include "knetworkmanager-devicestore.h"
#include "knetworkmanager-devicestore_dbus.h"
#include "knetworkmanager-network.h"
#include "knetworkmanager-encryption.h"
#include "knetworkmanager-synchronizer.h"

#include "kdebug.h"
#include "klocale.h"

KNetworkManager* DeviceStoreDBus::_ctx = NULL;

typedef struct {
	Device * dev;
	const char * active_net_path;
	const char * signal_name;
} UpdateCallbackData;

void
DeviceStoreDBus::activateDialUp (DialUp* dialup)
{
	DBusMessage*    msg = NULL;
	DBusConnection* con = _ctx->getDBus ()->getConnection ();

	if (!con || !dialup) {
		return;
	}

	msg = dbus_message_new_method_call (NM_DBUS_SERVICE, NM_DBUS_PATH, NM_DBUS_INTERFACE, "activateDialup");
	if (msg) {
		const char *name = dialup->getName (); 

		dbus_message_append_args (msg, DBUS_TYPE_STRING, &name, DBUS_TYPE_INVALID);
		dbus_connection_send (con, msg, NULL);
		dbus_message_unref (msg);
	}

	return;
}

void
DeviceStoreDBus::deactivateDialUp (DialUp* dialup)
{
	DBusMessage*    msg = NULL;
	DBusConnection* con = _ctx->getDBus ()->getConnection ();

	if (!con || !dialup) {
		return;
	}

	msg = dbus_message_new_method_call (NM_DBUS_SERVICE, NM_DBUS_PATH, NM_DBUS_INTERFACE, "deactivateDialup");
	if (msg) {
		const char *name = dialup->getName (); 

		dbus_message_append_args (msg, DBUS_TYPE_STRING, &name, DBUS_TYPE_INVALID);
		dbus_connection_send (con, msg, NULL);
		dbus_message_unref (msg);
	}
}

void
DeviceStoreDBus::activateDevice (Device* dev)
{
	DBusMessage*    msg = NULL;
	DBusConnection* con = _ctx->getDBus ()->getConnection ();

	if (!con) {
		return;
	}

	msg = dbus_message_new_method_call (NM_DBUS_SERVICE, NM_DBUS_PATH, NM_DBUS_INTERFACE, "setActiveDevice");
	if (msg) {
		const char *obj_path = dev->getObjectPath (); 

		dbus_message_append_args (msg, DBUS_TYPE_OBJECT_PATH, &obj_path, DBUS_TYPE_INVALID);
		dbus_connection_send (con, msg, NULL);
		dbus_message_unref (msg);
	}

	return;
}

void
DeviceStoreDBus::activateNetwork (Network* net, Device* dev)
{
	DBusMessage*    msg   = NULL;
	DBusConnection* con   = _ctx->getDBus ()->getConnection ();
	DeviceStore*    store = _ctx->getDeviceStore ();
	Encryption*     enc   = net->getEncryption ();
	const char *    essid = net->getEssid ();

	if (!con || !store || !essid) {
		return;
	}

	if (!dev) {
		dev = store->getDevice (net);
	}

	msg = dbus_message_new_method_call (NM_DBUS_SERVICE, NM_DBUS_PATH, NM_DBUS_INTERFACE, "setActiveDevice");
	if (msg) {
		const char* obj_path   = dev->getObjectPath ();
		const char* essid      = net->getEssid ();

		dbus_message_append_args (msg, DBUS_TYPE_OBJECT_PATH, &obj_path,
					       DBUS_TYPE_STRING,      &essid, DBUS_TYPE_INVALID);

		if (enc) {
			enc->serialize (msg, essid);
		}
		dbus_connection_send (con, msg, NULL);

		dbus_message_unref (msg);
	}

	return;
}

void
DeviceStoreDBus::getHalProperty (const QCString& udi, const QCString& property, QCString& result)
{
	DBusConnection* con      = _ctx->getDBus ()->getConnection ();
	LibHalContext*  hal_ctx  = NULL;
	char*           prop_val = NULL;

	if (!con || !property) {
		goto out;
	}

	if (!dbus_bus_name_has_owner (con, "org.freedesktop.Hal", NULL)) {
		printf ("Error: HAL seems not to be running.\n");
		goto out;
	}
	
	hal_ctx = libhal_ctx_new ();

	if (!libhal_ctx_set_dbus_connection (hal_ctx, con)) {
		goto out;
	}

	if (!libhal_ctx_init (hal_ctx, NULL)) {
		goto out;
	}

	prop_val = libhal_device_get_property_string (hal_ctx, udi, property, NULL);
	result = prop_val;
	libhal_free_string (prop_val);
out:
	if (hal_ctx) {
		libhal_ctx_shutdown (hal_ctx, NULL);
		libhal_ctx_free (hal_ctx);
	}

	return;
}

void
DeviceStoreDBus::setHalDeviceInfo (Device* dev)
{
	QCString device_udi = "";
	QCString parent_udi = "";
	QCString subsystem  = "";
	QCString vendor     = "";
	QCString product    = "";
	bustype  type       = BUS_UNKNOWN;

	device_udi = dev->getUdi ().utf8();
	getHalProperty (device_udi, "info.parent", parent_udi);
	getHalProperty (parent_udi, "linux.subsystem", subsystem);
	dev->setBustype(QString::fromUtf8( subsystem ) );

	type = dev->getBustype ();
	if (type == BUS_PCI || type == BUS_PCMCIA) {
		getHalProperty (parent_udi, "info.vendor", vendor);
		getHalProperty (parent_udi, "info.product", product);
	} else if (type == BUS_USB) {
		getHalProperty (parent_udi, "usb.vendor", vendor);
		getHalProperty (parent_udi, "usb.product", product);
	} else {
		getHalProperty (parent_udi, "info.vendor", vendor);
		getHalProperty (parent_udi, "info.product", product);
	}

	dev->setVendor (vendor.isNull () ? i18n ("Unknown") : QString::fromUtf8 (vendor));
	dev->setProduct (product.isNull() ? i18n ("Unknown") : QString::fromUtf8 (product));

	return;
}

void
DeviceStoreDBus::removeDevice (const char* obj_path)
{
	DeviceStore* store = _ctx->getDeviceStore ();
	Device* dev = store->getDevice (obj_path);

	if (dev) {
		store->emitRemoved (dev);
		store->removeDevice (dev);
		store->commitUpdate ();
	}

	return;
}

void
DeviceStoreDBus::removeNetwork (const char* obj_path, const char* net_path)
{
	DeviceStore* store = _ctx->getDeviceStore ();

	store->removeNetwork (obj_path, net_path);
	store->commitUpdate ();

	return;
}

void
DeviceStoreDBus::updateNetworkStrength (const char* obj_path, const char* net_path, int strength)
{
	DeviceStore* store = _ctx->getDeviceStore ();
	
	store->updateNetworkStrength (obj_path, net_path, strength);
	store->commitUpdate ();

	return;
}

void
DeviceStoreDBus::updateActivationStage (const char* obj_path, NMActStage act_stage)
{
	DeviceStore* store = _ctx->getDeviceStore ();

	store->updateActivationStage (obj_path, act_stage);
	store->commitUpdate ();

	return;
}

void
DeviceStoreDBus::updateNetworkCallback (DBusPendingCall* pcall, void* data)
{
	DBusMessage* reply = NULL;
	DeviceStore* store = _ctx->getDeviceStore ();
	UpdateCallbackData * cbData = (UpdateCallbackData *) data;
	Device*      dev   = cbData->dev;
	Network*     net   = NULL;

	const char*  obj_path     = NULL;
	const char*  essid        = NULL;
	const char*  hw_address   = NULL;
	dbus_int32_t strength     = -1;
	double       freq         = 0;
	dbus_int32_t rate         = 0;
	dbus_int32_t mode         = 0;
	dbus_int32_t capabilities = NM_802_11_CAP_NONE;
	dbus_bool_t  broadcast    = true;
	
	if (!dev || !pcall) {
		return;
	}

	reply = dbus_pending_call_steal_reply (pcall);
	if (!reply) {
		goto out;
	}

	if (dbus_message_is_error (reply, NM_DBUS_NO_NETWORKS_ERROR)) {
		dbus_message_unref (reply);
		goto out;
	}

	if (dbus_message_get_args (reply, NULL, DBUS_TYPE_OBJECT_PATH, &obj_path,
						DBUS_TYPE_STRING,      &essid,
						DBUS_TYPE_STRING,      &hw_address,
						DBUS_TYPE_INT32,       &strength,
						DBUS_TYPE_DOUBLE,      &freq,
						DBUS_TYPE_INT32,       &rate,
						DBUS_TYPE_INT32,       &mode,
						DBUS_TYPE_INT32,       &capabilities,
						DBUS_TYPE_BOOLEAN,     &broadcast, DBUS_TYPE_INVALID)) {
		Synchronizer sync(dev);
		//TODO: Aare these props really UTF-8? I thought they are Latin-1. It could save a pair of CPU cycles for us ;-)
		net = sync.synchronize(QString::fromUtf8(essid), QString::fromUtf8(obj_path));

		if (cbData->active_net_path)
			if (strcmp (obj_path, cbData->active_net_path) == 0) {
				net->setActive (true);
			} else {
				net->setActive (false);
			}

		//net->setObjectPath (obj_path);
		net->setEssid (QString::fromUtf8(essid));
		net->insertHardwareAddress (hw_address, true);
		net->setStrength (strength);
		net->setFrequency (freq);
		net->setRate (rate);
		net->setMode (mode);
		net->setCapabilities (capabilities);
		net->setHidden (!broadcast);
		
		store->commitUpdate ();
		if ( !qstrcmp( cbData->signal_name, "WirelessNetworkAppeared" ) )
			store->emitNetworkFound( net );
	}

	dbus_message_unref (reply);

out:
	delete cbData;
	dbus_pending_call_unref (pcall);

	return;
}

void
DeviceStoreDBus::updateNetwork (const char* obj_path, const char* net_path, const char* active_net_path, const char * signal_name )
{
	DBusMessage*        msg    = NULL;
	DBusConnection*     con    = _ctx->getDBus ()->getConnection ();
	DBusPendingCall*    pcall  = NULL;
	DeviceStore*        store  = _ctx->getDeviceStore ();
	Device*             dev    = NULL;
	UpdateCallbackData* cbData = 0;
	
	if (!obj_path || !net_path || !con || !store) {
		return;
	}

	dev = store->getDevice (obj_path);
	if (!dev) {
		printf ("updateNetwork: Found network without device? Bailing out.\n");
		return;
	}

	msg = dbus_message_new_method_call (NM_DBUS_SERVICE, net_path, NM_DBUS_INTERFACE_DEVICES, "getProperties");
	if (msg) {
		cbData = new UpdateCallbackData; 
		cbData->dev = dev;
		cbData->active_net_path = active_net_path;
		cbData->signal_name = signal_name;

		dbus_connection_send_with_reply (con, msg, &pcall, -1);
		if (pcall) {
			dbus_pending_call_set_notify (pcall, DeviceStoreDBus::updateNetworkCallback, cbData, NULL);
		}
		dbus_message_unref (msg);
	}
}

void
DeviceStoreDBus::updateDeviceCallback (DBusPendingCall* pcall, void* data )
{
	DBusMessage* reply = NULL;
	DeviceStore* store = _ctx->getDeviceStore ();
	Device*      dev   = NULL;
	const char*  signal_name = (const char *) data;

	const char*   obj_path          = NULL;
	const char*   interface         = NULL;
	NMDeviceType  type              = DEVICE_TYPE_UNKNOWN;
	const char*   udi               = NULL;
	dbus_bool_t   active            = false;
	NMActStage    act_stage         = NM_ACT_STAGE_UNKNOWN;
	const char*   ipv4_address      = NULL;
	const char*   subnetmask        = NULL;
	const char*   broadcast         = NULL;
	const char*   hw_address        = NULL;
	const char*   route             = NULL;
	const char*   pri_dns           = NULL;
	const char*   sec_dns           = NULL;
	dbus_int32_t  mode              = 0;
	dbus_int32_t  strength          = -1;
	dbus_bool_t   link_active       = false;
	dbus_int32_t  speed             = 0;
	dbus_uint32_t capabilities      = NM_DEVICE_CAP_NONE;
	dbus_uint32_t capabilities_type = NM_DEVICE_CAP_NONE;
	char**        networks          = NULL;
	int           num_networks      = 0;
	const char*   active_net_path   = NULL;

	if (!pcall) {
		return;
	}

	reply = dbus_pending_call_steal_reply (pcall);
	if (!reply) {
		goto out;
	}

	if (dbus_message_is_error (reply, NM_DBUS_NO_NETWORKS_ERROR)) {
		dbus_message_unref (reply);
		goto out;
	}

	if (dbus_message_get_args (reply, NULL, DBUS_TYPE_OBJECT_PATH,             &obj_path,
						DBUS_TYPE_STRING,                  &interface,
						DBUS_TYPE_UINT32,                  &type,
						DBUS_TYPE_STRING,                  &udi,
						DBUS_TYPE_BOOLEAN,                 &active,
						DBUS_TYPE_UINT32,                  &act_stage,
						DBUS_TYPE_STRING,                  &ipv4_address,
						DBUS_TYPE_STRING,                  &subnetmask,
						DBUS_TYPE_STRING,                  &broadcast,
						DBUS_TYPE_STRING,                  &hw_address,
						DBUS_TYPE_STRING,                  &route,
						DBUS_TYPE_STRING,                  &pri_dns,
						DBUS_TYPE_STRING,                  &sec_dns,
						DBUS_TYPE_INT32,                   &mode,
						DBUS_TYPE_INT32,                   &strength,
						DBUS_TYPE_BOOLEAN,                 &link_active,
						DBUS_TYPE_INT32,                   &speed,
						DBUS_TYPE_UINT32,                  &capabilities,
						DBUS_TYPE_UINT32,                  &capabilities_type,
						DBUS_TYPE_STRING,                  &active_net_path,
						DBUS_TYPE_ARRAY, DBUS_TYPE_STRING, &networks, &num_networks, DBUS_TYPE_INVALID)) {
		dev = store->getDevice (obj_path);
		if (dev == NULL) {
			store->addDevice (new Device (obj_path));
			dev = store->getDevice (obj_path);
		}

		if (active == true) {
			store->invalidateActiveDevices ();
		}

		dev->setInterface (interface);
		dev->setObjectPath (obj_path);
		dev->setType (type);
		dev->setUdi (udi);
		dev->setActive (active);
		dev->setActivationStage (act_stage);
		dev->setIPv4Address (ipv4_address);
		dev->setSubnetmask (subnetmask);
		dev->setBroadcast (broadcast);
		dev->setHardwareAddress (hw_address);
		dev->setRoute (route);
		dev->setPrimaryDNS (pri_dns);
		dev->setSecondaryDNS (sec_dns);
		dev->setMode (mode);
		dev->setStrength (strength);
		dev->setLinkActive (link_active);
		dev->setSpeed (speed);
		dev->setCapabilities (capabilities);
		dev->setCapabilitiesType (capabilities_type);

		setHalDeviceInfo (dev);

		if ((dev->isWireless()) && (num_networks > 0)) {
			char** net_path;
			
			for (net_path = networks; *net_path; net_path++) {
				DeviceStoreDBus::updateNetwork (obj_path, *net_path, active_net_path);
			}
		}
		dbus_free_string_array (networks);

		store->commitUpdate ();
		if ( signal_name )
		{
			if ( !qstrcmp( signal_name, "DeviceStrengthChanged" ) )
				store->emitStrengthChange( dev );
			else if ( !qstrcmp( signal_name, "DeviceCarrierOn" ) )
				store->emitCarrierOn( dev );
			else if ( !qstrcmp( signal_name, "DeviceCarrierOff" ) )
				store->emitCarrierOff( dev );
			else if ( !qstrcmp( signal_name, "DeviceAdded" ) )
				store->emitAdded( dev );
			else if ( !qstrcmp( signal_name, "DeviceNoLongerActive" ) )
				store->emitNoLongerActive( dev );
			else if ( !qstrcmp( signal_name, "DeviceNowActive" ) )
				store->emitActive( dev );
			else if ( !qstrcmp( signal_name, "DeviceActivating" ) )
				store->emitActivating( dev );
		}
	} else {
		printf ("error updating device\n");
	}
	dbus_message_unref (reply);

out:
	dbus_pending_call_unref (pcall);

	return;
}

void
DeviceStoreDBus::updateDevice (const char* obj_path, const char * signal_name)
{
	DBusMessage*     msg   = NULL;
	DBusConnection*  con   = _ctx->getDBus()->getConnection ();
	DBusPendingCall* pcall = NULL;
	
	if (!con|| !obj_path) {
		return;
	}

	msg = dbus_message_new_method_call (NM_DBUS_SERVICE, obj_path, NM_DBUS_INTERFACE_DEVICES, "getProperties");
	if (msg) {
		dbus_connection_send_with_reply (con, msg, &pcall, -1);
		if (pcall)  {
			dbus_pending_call_set_notify (pcall, DeviceStoreDBus::updateDeviceCallback, (char *)signal_name, NULL);
		}
		dbus_message_unref (msg);
	}

	return;
}

void DeviceStoreDBus::clearStore ()
{
	DeviceStore* store = _ctx->getDeviceStore ();

	store->clear ();
	store->commitUpdate ();
}

void
DeviceStoreDBus::getDialUpCallback (DBusPendingCall* pcall, void* /*data*/)
{
	DBusMessage* reply = NULL;
	DeviceStore* store = _ctx->getDeviceStore ();

	char** dialups     = NULL;
	int    num_dialups = 0;

	reply = dbus_pending_call_steal_reply (pcall);
	if (reply == NULL) {
		goto out;
	}

	if (dbus_message_is_error (reply, NM_DBUS_NO_DIALUP_ERROR)) {
		dbus_message_unref (reply);
		goto out;
	}

	if (dbus_message_get_args (reply, NULL, DBUS_TYPE_ARRAY, DBUS_TYPE_STRING, &dialups, &num_dialups, DBUS_TYPE_INVALID)) {
		for (char** item = dialups; *item; item++) {
			store->addDialUp (new DialUp (*item));
		}
		dbus_free_string_array (dialups);
	}
	dbus_message_unref (reply);

	store->commitUpdate ();

out:
	dbus_pending_call_unref (pcall);

	return;
}

void
DeviceStoreDBus::getDialUp ()
{
	DBusMessage*     msg    = NULL;
	DBusConnection*  con    = _ctx->getDBus()->getConnection ();
	DBusPendingCall* pcall  = NULL;

	if (!con) {
		return;
	}

	msg = dbus_message_new_method_call (NM_DBUS_SERVICE, NM_DBUS_PATH, NM_DBUS_INTERFACE_DEVICES, "getDialup");
	if (msg) {
		dbus_connection_send_with_reply (con, msg, &pcall, -1);
		if (pcall) {
			dbus_pending_call_set_notify (pcall, DeviceStoreDBus::getDialUpCallback, NULL, NULL);
		}
		dbus_message_unref (msg);
	}

	return;
}

void
DeviceStoreDBus::getDevicesCallback (DBusPendingCall* pcall, void* /*data*/)
{
	DBusMessage* reply       = NULL;
	char**       devices     = NULL;
	int          num_devices = 0;

	reply = dbus_pending_call_steal_reply (pcall);
	if (reply == NULL) {
		goto out;
	}

	if (dbus_message_is_error (reply, NM_DBUS_NO_DEVICES_ERROR)) {
		dbus_message_unref (reply);
		goto out;
	}

	if (dbus_message_get_args (reply, NULL, DBUS_TYPE_ARRAY, DBUS_TYPE_OBJECT_PATH, &devices, &num_devices, DBUS_TYPE_INVALID)) {
		for (char** obj_path = devices; *obj_path; obj_path++) {
			DeviceStoreDBus::updateDevice (*obj_path);
		}
		dbus_free_string_array (devices);
	}
	dbus_message_unref (reply);

out:
	dbus_pending_call_unref (pcall);

	return;
}

void
DeviceStoreDBus::getDevices ()
{
	DBusMessage*     msg    = NULL;
	DBusConnection*  con    = _ctx->getDBus()->getConnection ();
	DBusPendingCall* pcall  = NULL;
	
	if (!con) {
		return;
	}

	msg = dbus_message_new_method_call (NM_DBUS_SERVICE, NM_DBUS_PATH, NM_DBUS_INTERFACE_DEVICES, "getDevices");
	if (msg) {
		dbus_connection_send_with_reply (con, msg, &pcall, -1);
		if (pcall) {
			dbus_pending_call_set_notify (pcall, DeviceStoreDBus::getDevicesCallback, NULL, NULL);
		}
		dbus_message_unref (msg);
	}

	return;
}

void
DeviceStoreDBus::populateStore ()
{
	DeviceStoreDBus::getDevices ();
	DeviceStoreDBus::getDialUp  ();

	return;
}

void
DeviceStoreDBus::push (KNetworkManager* ctx)
{
	_ctx = ctx;
}
