/***************************************************************************
 *   Copyright (C) 2008 by Peter Penz <peter.penz19@gmail.com>             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA            *
 ***************************************************************************/

#ifndef DOLPHINFILEITEMDELEGATE_H
#define DOLPHINFILEITEMDELEGATE_H

#include <KFileItemDelegate>
#include <views/dolphinmodel.h>

class QAbstractProxyModel;

/**
 * Extends KFileItemDelegate by the ability to show the hover effect
 * and the selection in a minimized way for the name column of
 * the details view.
 *
 * Note that this is a workaround, as Qt does not support having custom
 * shapes within the visual rect of an item view. The visual part of
 * workaround is handled inside DolphinFileItemDelegate, the behavior
 * changes are handled in DolphinTreeView.
 */
class DolphinFileItemDelegate : public KFileItemDelegate
{
    Q_OBJECT
public:
    explicit DolphinFileItemDelegate(QObject* parent = 0);
    virtual ~DolphinFileItemDelegate();

    /**
     * If \a minimized is true, the hover effect and the selection are
     * only drawn above the icon and text of an item. Per default
     * \a minimized is false, which means that the whole visual rect is
     * used like in KFileItemDelegate.
     */
    void setMinimizedNameColumn(bool minimized);
    bool hasMinimizedNameColumn() const;

    virtual void paint(QPainter* painter,
                       const QStyleOptionViewItem& option,
                       const QModelIndex& index) const;

    /**
     * Returns the minimized width of the name column for the name \a name. This method
     * is also used in DolphinDetailsView to handle the selection of items correctly.
     */
    static int nameColumnWidth(const QString& name, const QStyleOptionViewItem& option);

private slots:
    void handleDisplayPaletteChange();

private:
    static void adjustOptionWidth(QStyleOptionViewItemV4& option,
                                  const QAbstractProxyModel* proxyModel,
                                  const DolphinModel* dolphinModel,
                                  const QModelIndex& index);

    static void adjustOptionTextColor(QStyleOptionViewItemV4& option,
                                      KVersionControlPlugin::VersionState state);

    QPixmap emblemForState(KVersionControlPlugin::VersionState state, const QSize& size) const;

private:
    bool m_hasMinimizedNameColumn;
    mutable QSize m_cachedSize;
    mutable QPixmap m_cachedEmblems[KVersionControlPlugin::LocallyModifiedUnstagedVersion + 1];
    mutable QColor m_cachedInactiveTextColor;
    mutable bool m_cachedInactiveTextColorDirty;
};

inline void DolphinFileItemDelegate::setMinimizedNameColumn(bool minimized)
{
    m_hasMinimizedNameColumn = minimized;
}

inline bool DolphinFileItemDelegate::hasMinimizedNameColumn() const
{
    return m_hasMinimizedNameColumn;
}

#endif
