/* This file is part of the KDE project
   Copyright (C) 2001 Carsten Pfeiffer <pfeiffer@kde.org>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <dcopclient.h>

#include <kapp.h>
#include <kaction.h>
#include <kconfig.h>
#include <kglobal.h>
#include <khtml_part.h>
#include <kinstance.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>
#include <kprotocolmanager.h>

#include "settingsplugin.h"

SettingsPlugin::SettingsPlugin( QObject* parent, const char* name )
    : KParts::Plugin( parent, name )
{
    if ( !kapp->dcopClient()->isAttached() )
	kapp->dcopClient()->attach();

    KActionMenu *menu = new KActionMenu( i18n("HTML Settings"), "configure",
					 actionCollection(), "action menu" );
    menu->setDelayed( false );

    KToggleAction *action;
    action = new KToggleAction( i18n("Java&script"), 0,
				this, SLOT(toggleJavascript()),
				actionCollection(), "javascript" );
    menu->insert( action );
	
    action = new KToggleAction( i18n("&Java"), 0,
				this, SLOT(toggleJava()),
				actionCollection(), "java" );
    menu->insert( action );

    action = new KToggleAction( i18n("&Cookies"), 0,
				this, SLOT(toggleCookies()),
				actionCollection(), "cookies" );
    menu->insert( action );

    action = new KToggleAction( i18n("&Plugins"), 0,
				this, SLOT(togglePlugins()),
				actionCollection(), "plugins" );
    menu->insert( action );

    action = new KToggleAction( i18n("Autoload &Images"), 0,
				this, SLOT(toggleImageLoading()),
				actionCollection(), "imageloading" );
    menu->insert( action );

    action = new KToggleAction( i18n("Enable Cache"), 0,
				this, SLOT(toggleCache()),
				actionCollection(), "usecache" );
    menu->insert( action );

    KSelectAction *sAction = new KSelectAction( i18n("Cache Policy"), 0,
						0, 0,
						actionCollection(), "cachepolicy" );
    QStringList policies;
    policies += i18n( "&Keep Cache In Sync" );
    policies += i18n( "&Use Cache If Possible" );
    policies += i18n( "&Offline Browsing Mode" );
    sAction->setItems( policies );
    connect( sAction, SIGNAL( activated( int ) ), SLOT( cachePolicyChanged(int) ) );

    menu->insert( sAction );

    connect( menu->popupMenu(), SIGNAL( aboutToShow() ), SLOT( showPopup() ));
}

SettingsPlugin::~SettingsPlugin()
{
}

void SettingsPlugin::showPopup()
{
    KHTMLPart *part = dynamic_cast<KHTMLPart *>( parent() );
    if ( !part )
	return;

    bool cookies = cookiesEnabled( part->url().url() );

    ((KToggleAction*)actionCollection()->action("java"))->setChecked( part->javaEnabled() );
    ((KToggleAction*)actionCollection()->action("javascript"))->setChecked( part->jScriptEnabled() );
    ((KToggleAction*)actionCollection()->action("cookies"))->setChecked( cookies );
    ((KToggleAction*)actionCollection()->action("plugins"))->setChecked( part->pluginsEnabled() );
    ((KToggleAction*)actionCollection()->action("imageloading"))->setChecked( part->autoloadImages() );
    ((KToggleAction*)actionCollection()->action("usecache"))->setChecked(KProtocolManager::useCache());

    KIO::CacheControl cc = KProtocolManager::cacheControl();
    switch ( cc ) {
    case KIO::CC_Verify:
      ((KSelectAction*)actionCollection()->action("cachepolicy"))->setCurrentItem( 0 );
      break;
    case KIO::CC_CacheOnly:
      ((KSelectAction*)actionCollection()->action("cachepolicy"))->setCurrentItem( 2 );
      break;
    case KIO::CC_Cache:
      ((KSelectAction*)actionCollection()->action("cachepolicy"))->setCurrentItem( 1 );
      break;
    };
}

void SettingsPlugin::toggleJava()
{
    KHTMLPart *part = dynamic_cast<KHTMLPart *>( parent() );
    if ( part ) {
        part->enableJava( ((KToggleAction*)actionCollection()->action("java"))->isChecked() );
    }
}

void SettingsPlugin::toggleJavascript()
{
    KHTMLPart *part = dynamic_cast<KHTMLPart *>( parent() );
    if ( part ) {
        part->enableJScript( ((KToggleAction*)actionCollection()->action("javascript"))->isChecked() );
    }
}

void SettingsPlugin::toggleCookies()
{
    KHTMLPart *part = dynamic_cast<KHTMLPart *>( parent() );
    if ( !part )
	return;

    QString advice;
    bool enable = ((KToggleAction*)actionCollection()->action("cookies"))->isChecked();
    advice = enable ? "Accept" : "Reject";

    QCString replyType;
    QByteArray data, replyData;
    QDataStream stream( data, IO_WriteOnly );
    stream << part->url().url() << advice;
    bool ok = kapp->dcopClient()->call( "kcookiejar", "kcookiejar",
					"setDomainAdvice(QString,QString)",
					data, replyType, replyData, true );

    // cookies disabled completely -> we have to start kcookiejar
    if ( !ok && enable ) {
	ok = ( kapp->startServiceByDesktopName("kcookiejar") == 0 );
	if ( ok )
	    ok = kapp->dcopClient()->call( "kcookiejar", "kcookiejar",
					   "setDomainAdvice(QString,QString)",
					   data, replyType, replyData, true );
	
	if ( !ok )
	    KMessageBox::sorry( part->widget(),
				i18n("I can't enable cookies, because the\n"
				     "cookie daemon could not be started."),
				i18n("Cookies disabled"));
    }
}

void SettingsPlugin::togglePlugins()
{
    KHTMLPart *part = dynamic_cast<KHTMLPart *>( parent() );
    if ( part ) {
        part->enablePlugins( ((KToggleAction*)actionCollection()->action("plugins"))->isChecked() );
    }
}

void SettingsPlugin::toggleImageLoading()
{
    KHTMLPart *part = dynamic_cast<KHTMLPart *>( parent() );
    if ( part ) {
        part->autoloadImages( ((KToggleAction*)actionCollection()->action("imageloading"))->isChecked() );
    }
}

bool SettingsPlugin::cookiesEnabled( const QString& url )
{
    QByteArray data, reply;
    QCString replyType;
    QDataStream stream( data, IO_WriteOnly );
    stream << url;
    kapp->dcopClient()->call( "kcookiejar", "kcookiejar", "getDomainAdvice(QString)", data, replyType, reply, true );

    bool enabled = false;

    if ( replyType == "QString" ) {
	QString advice;
	QDataStream s( reply, IO_ReadOnly );
	s >> advice;
	enabled = ( advice == "Accept" );
	if ( !enabled && advice == "Dunno" ) {
	    // TODO, check the global setting via dcop
	    KConfig kc( "kcookiejarrc", true, false );
	    kc.setGroup( "Cookie Policy" );
	    enabled =
		(kc.readEntry( "CookieGlobalAdvice", "Reject" ) == "Accept");
	}
    }

    return enabled;
}

void SettingsPlugin::toggleCache()
{
  KProtocolManager::setUseCache( !KProtocolManager::useCache() );
  ((KToggleAction*)actionCollection()->action("usecache"))->setChecked(KProtocolManager::useCache());
}

void SettingsPlugin::cachePolicyChanged( int p )
{
  switch ( p ) {
  case 0:
    KProtocolManager::setCacheControl( KIO::CC_Verify );
    break;
  case 1:
    KProtocolManager::setCacheControl( KIO::CC_Cache );
    break;
  case 2:
    KProtocolManager::setCacheControl( KIO::CC_CacheOnly );
    break;
  };
}

///////////////////////////////////////////////////////////////////


KInstance* KHTMLSettingsFactory::s_instance = 0L;

KHTMLSettingsFactory::KHTMLSettingsFactory( QObject* parent, const char* name )
  : KLibFactory( parent, name )
{
    s_instance = new KInstance("KHTMLSettingsFactory");
    KGlobal::locale()->insertCatalogue( QString::fromLatin1("khtmlsettingsplugin") );
}

KHTMLSettingsFactory::~KHTMLSettingsFactory()
{
    delete s_instance;
    s_instance = 0L;
}

QObject* KHTMLSettingsFactory::createObject( QObject* parent, const char* name, const char*, const QStringList & )
{
    return new SettingsPlugin( parent, name );
}

extern "C"
{
  void * init_libkhtmlsettingsplugin()
  {
    return new KHTMLSettingsFactory;
  }
}

#include "settingsplugin.moc"
