// graph-tool -- a general graph modification and manipulation thingy
//
// Copyright (C) 2006-2024 Tiago de Paula Peixoto <tiago@skewed.de>
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU Lesser General Public License as published by the Free
// Software Foundation; either version 3 of the License, or (at your option) any
// later version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "graph_filtering.hh"
#include "graph.hh"
#include "graph_properties.hh"

#include <boost/graph/metric_tsp_approx.hpp>

using namespace std;
using namespace boost;
using namespace graph_tool;

struct get_tsp_approx
{
    template <class Graph, class WeightMap, class IntType>
    void operator()(Graph& g, size_t src, WeightMap weights,
                    vector<IntType>& tour) const
    {
        back_insert_iterator<vector<IntType> > back_it(tour);
        metric_tsp_approx_tour_from_vertex(g, vertex(src, g), weights,
                                           back_it);
    }
};

vector<int32_t> get_tsp(GraphInterface& gi, size_t src, boost::any weight_map)
{
    vector<int32_t> tour;

    typedef UnityPropertyMap<size_t,GraphInterface::edge_t> cweight_t;

    if (weight_map.empty())
        weight_map = cweight_t();

    typedef mpl::push_back<edge_scalar_properties, cweight_t>::type
        weight_maps;

    run_action<graph_tool::detail::never_directed>()
        (gi,
         [&](auto&& graph, auto&& a2)
         {
             return get_tsp_approx()
                 (std::forward<decltype(graph)>(graph), src,
                  std::forward<decltype(a2)>(a2), tour);
         },
         weight_maps())(weight_map);

    return tour;
}

#include <boost/python.hpp>

using namespace boost::python;

#define __MOD__ topology
#include "module_registry.hh"
REGISTER_MOD
([]
 {
     def("get_tsp", &get_tsp);
 });
