const char uri_lua[] =
"-- uri.lua (internal file)\n"
"\n"
"local ffi = require('ffi')\n"
"local buffer = require('buffer')\n"
"local internal = require('uri.lib')\n"
"\n"
"local uri_cdef = [[\n"
"struct uri_param {\n"
"    const char *name;\n"
"    int values_capacity;\n"
"    int value_count;\n"
"    const char **values;\n"
"};\n"
"\n"
"/**\n"
" * We define all strings inside the `struct uri` as const, despite\n"
" * the fact that they are not constant in the C structure. This is\n"
" * necessary for the `uri_format` function work: there we cannot\n"
" * assign lua strings to a non-constant pointer. It's not a problem\n"
" * since uri_format doesn't change something in `struct uri`.\n"
" */\n"
"struct uri {\n"
"    const char *scheme;\n"
"    const char *login;\n"
"    const char *password;\n"
"    const char *host;\n"
"    const char *service;\n"
"    const char *path;\n"
"    const char *query;\n"
"    const char *fragment;\n"
"    int host_hint;\n"
"    int params_capacity;\n"
"    int param_count;\n"
"    struct uri_param *params;\n"
"};\n"
"\n"
"struct uri_set {\n"
"    int uri_count;\n"
"    struct uri *uris;\n"
"};\n"
"\n"
"void\n"
"uri_destroy(struct uri *uri);\n"
"\n"
"void\n"
"uri_set_destroy(struct uri_set *uri_set);\n"
"\n"
"int\n"
"uri_format(char *str, size_t len, struct uri *uri, bool write_password);\n"
"\n"
"size_t\n"
"uri_escape(const char *src, size_t src_size, char *dst,\n"
"           const unsigned char unreserved[256], bool encode_plus);\n"
"\n"
"size_t\n"
"uri_unescape(const char *src, size_t src_size, char *dst, bool decode_plus);\n"
"]]\n"
"\n"
"pcall(ffi.cdef, uri_cdef) -- Required for running unit tests.\n"
"\n"
"local builtin = ffi.C;\n"
"local uri_stash = buffer.ffi_stash_new('struct uri')\n"
"local uri_stash_take = uri_stash.take\n"
"local uri_stash_put = uri_stash.put\n"
"local cord_ibuf_take = buffer.internal.cord_ibuf_take\n"
"local cord_ibuf_put = buffer.internal.cord_ibuf_put\n"
"\n"
"local uri_set_stash = buffer.ffi_stash_new('struct uri_set')\n"
"local uri_set_stash_take = uri_set_stash.take\n"
"local uri_set_stash_put = uri_set_stash.put\n"
"\n"
"local function unreserved(str)\n"
"    if type(str) ~= \"string\" then\n"
"        error(\"Usage: uri.unreserved(str)\")\n"
"    end\n"
"    local pattern = \"^[\" .. str .. \"]$\"\n"
"    local unreserved_tbl = ffi.new(\"unsigned char[256]\")\n"
"    for i = 0, 255 do\n"
"        -- By default symbol is reserved.\n"
"        unreserved_tbl[i] = 0\n"
"        -- i-th symbol in the extended ASCII table.\n"
"        local ch = string.char(i)\n"
"        if ch:match(pattern) then\n"
"            unreserved_tbl[i] = 1\n"
"        end\n"
"    end\n"
"    return unreserved_tbl\n"
"end\n"
"\n"
"-- Some characters, called magic characters, have special meanings when used in\n"
"-- a pattern. The magic characters are: % - ^\n"
"-- These characters must be escaped in patterns with unreserved symbols.\n"
"\n"
"local ALPHA = \"abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ\"\n"
"local DIGIT = \"0123456789\"\n"
"local ALNUM = ALPHA .. DIGIT\n"
"\n"
"local RFC3986 = {\n"
"    unreserved = unreserved(ALNUM .. \"%-._~\"),\n"
"    plus = false,\n"
"}\n"
"\n"
"local PATH = {\n"
"    unreserved = unreserved(ALNUM .. \"%-._~\" .. \"!$&'()*+,;=\" .. \"/\" .. \":@\"),\n"
"    plus = false,\n"
"}\n"
"\n"
"local PATH_PART = {\n"
"    unreserved = unreserved(ALNUM .. \"%-._~\" .. \"!$&'()*+,;=\" .. \":@\"),\n"
"    plus = false,\n"
"}\n"
"\n"
"local QUERY = {\n"
"    unreserved = unreserved(ALNUM .. \"%-._~\" .. \"!$&'()*+,;=\" .. \"/\?\"),\n"
"    plus = false,\n"
"}\n"
"\n"
"local QUERY_PART = {\n"
"    unreserved = unreserved(ALNUM .. \"%-._~\" .. \"!$'()*+,;\" .. \"/\?\"),\n"
"    plus = false,\n"
"}\n"
"\n"
"local FRAGMENT = {\n"
"    unreserved = unreserved(ALNUM .. \"%-._~\" .. \"!$&'()*+,;=\" .. \"/\?\"),\n"
"    plus = false,\n"
"}\n"
"\n"
"local FORM_URLENCODED = {\n"
"    unreserved = unreserved(ALNUM .. \"%-._\" .. \"*\"),\n"
"    plus = true,\n"
"}\n"
"\n"
"local function parse_uribuf(uribuf)\n"
"    local result = {}\n"
"    for _, k in ipairs({ 'scheme', 'login', 'password', 'host', 'service',\n"
"        'path', 'query', 'fragment'}) do\n"
"        if uribuf[k] ~= nil then\n"
"            result[k] = ffi.string(uribuf[k])\n"
"        end\n"
"    end\n"
"    if uribuf.param_count > 0 then\n"
"        result.params = {}\n"
"        for param_idx = 0, uribuf.param_count - 1 do\n"
"            local param = uribuf.params[param_idx]\n"
"            local name = ffi.string(param.name)\n"
"            result.params[name] = {}\n"
"            for val_idx = 0, param.value_count - 1 do\n"
"                result.params[name][val_idx + 1] =\n"
"                    ffi.string(param.values[val_idx])\n"
"            end\n"
"        end\n"
"    end\n"
"    if uribuf.host_hint == 1 then\n"
"        result.ipv4 = result.host\n"
"    elseif uribuf.host_hint == 2 then\n"
"        result.ipv6 = result.host\n"
"    elseif uribuf.host_hint == 3 then\n"
"        result.unix = result.service\n"
"    end\n"
"    return result\n"
"end\n"
"\n"
"local function parse(str)\n"
"    if str == nil then\n"
"        error(\"Usage: uri.parse(string|table)\")\n"
"    end\n"
"    local uribuf = uri_stash_take()\n"
"    local status, errmsg = pcall(internal.uri_create, uribuf, str)\n"
"    if not status then\n"
"        uri_stash_put(uribuf)\n"
"        return nil, errmsg\n"
"    end\n"
"    local result = parse_uribuf(uribuf)\n"
"    builtin.uri_destroy(uribuf)\n"
"    uri_stash_put(uribuf)\n"
"    return result\n"
"end\n"
"\n"
"local function parse_many(str)\n"
"    if str == nil then\n"
"        error(\"Usage: uri.parse_many(string|table)\")\n"
"    end\n"
"    local uri_set_buf = uri_set_stash_take()\n"
"    local status, errmsg = pcall(internal.uri_set_create, uri_set_buf, str)\n"
"    if not status then\n"
"        uri_set_stash_put(uri_set_buf)\n"
"        return nil, errmsg\n"
"    end\n"
"    local result = {}\n"
"    for i = 0, uri_set_buf.uri_count - 1 do\n"
"        result[i + 1] = parse_uribuf(uri_set_buf.uris[i])\n"
"    end\n"
"    builtin.uri_set_destroy(uri_set_buf)\n"
"    uri_set_stash_put(uri_set_buf)\n"
"    return result\n"
"end\n"
"\n"
"local function fill_uribuf_params(uribuf, uri)\n"
"    uribuf.param_count = 0\n"
"    uribuf.params = nil\n"
"    if not uri.params then\n"
"        return\n"
"    end\n"
"    for _, _ in pairs(uri.params) do\n"
"        uribuf.param_count = uribuf.param_count + 1\n"
"    end\n"
"    if uribuf.param_count == 0 then\n"
"        return\n"
"    end\n"
"    uribuf.params = ffi.new(\"struct uri_param[\?]\", uribuf.param_count)\n"
"    local i = 0\n"
"    for param_name, param in pairs(uri.params) do\n"
"        uribuf.params[i].name = param_name\n"
"        if type(param) == \"table\" then\n"
"            uribuf.params[i].value_count = #param\n"
"            uribuf.params[i].values = ffi.new(\"const char *[\?]\", #param)\n"
"            for j = 1, uribuf.params[i].value_count do\n"
"                uribuf.params[i].values[j - 1] = param[j]\n"
"            end\n"
"        elseif param == nil then\n"
"            uribuf.params[i].value_count = 0\n"
"        else\n"
"            uribuf.params[i].value_count = 1\n"
"            uribuf.params[i].values = ffi.new(\"const char *[\?]\", 1)\n"
"            uribuf.params[i].values[0] = param\n"
"        end\n"
"        i = i + 1\n"
"    end\n"
"end\n"
"\n"
"local function format(uri, write_password)\n"
"    local uribuf = uri_stash_take()\n"
"    uribuf.scheme = uri.scheme\n"
"    uribuf.login = uri.login\n"
"    uribuf.password = uri.password\n"
"    uribuf.host = uri.host\n"
"    uribuf.service = uri.service\n"
"    uribuf.path = uri.path\n"
"    uribuf.query = uri.query\n"
"    uribuf.fragment = uri.fragment\n"
"    -- The `uri_format` function needs the hint to enclose an IPv6\n"
"    -- address into square brackets. So we must set the `host_hint`\n"
"    -- to determine the type of `uri.host`.\n"
"    if uri.ipv4 ~= nil then\n"
"        uribuf.host_hint = 1\n"
"    elseif uri.ipv6 ~= nil then\n"
"        uribuf.host_hint = 2\n"
"    else\n"
"        uribuf.host_hint = 3\n"
"    end\n"
"    fill_uribuf_params(uribuf, uri)\n"
"    local ibuf = cord_ibuf_take()\n"
"    local str = ibuf:alloc(1024)\n"
"    local len = builtin.uri_format(str, 1024, uribuf,\n"
"                                   write_password and 1 or 0)\n"
"    uri_stash_put(uribuf)\n"
"    str = ffi.string(str, len)\n"
"    cord_ibuf_put(ibuf)\n"
"    return str\n"
"end\n"
"\n"
"local function build_opts(opts)\n"
"    if opts and type(opts) ~= \"table\" then\n"
"        error(\"opts must be a table\")\n"
"    end\n"
"    -- Attention: setting RFC3986.plus to true will break expression with\n"
"    -- choosing user and default value below.\n"
"    assert(RFC3986.plus == false)\n"
"    local options = {\n"
"        unreserved = (opts and opts.unreserved) or RFC3986.unreserved,\n"
"        plus = (opts and opts.plus) or RFC3986.plus,\n"
"    }\n"
"    if options.unreserved ~= nil and type(options.unreserved) ~= \"cdata\" then\n"
"        error(\"use uri.unreserved for building opts.unreserved\")\n"
"    end\n"
"    if type(options.plus) ~= \"boolean\" then\n"
"        error(\"opts.plus must be a boolean\")\n"
"    end\n"
"    return options\n"
"end\n"
"\n"
"-- Encodes a string into its escaped hexadecimal representation.\n"
"local function escape(buf, opts)\n"
"    if type(buf) ~= \"string\" then\n"
"        error(\"Usage: uri.escape(string, [opts])\")\n"
"    end\n"
"    local options = build_opts(opts)\n"
"\n"
"    -- The worst case is when all characters are encoded.\n"
"    local dst = ffi.new(\"char[\?]\", #buf * 3)\n"
"    local dst_size = builtin.uri_escape(buf, #buf, dst,\n"
"                                        options.unreserved,\n"
"                                        options.plus)\n"
"    return ffi.string(dst, dst_size)\n"
"end\n"
"\n"
"-- Decodes an escaped hexadecimal string into its binary representation.\n"
"local function unescape(buf, opts)\n"
"    if type(buf) ~= \"string\" then\n"
"        error(\"Usage: uri.unescape(string, [opts])\")\n"
"    end\n"
"    local options = build_opts(opts)\n"
"\n"
"    -- The worst case is when all characters were not decoded.\n"
"    local dst = ffi.new(\"char[\?]\", #buf)\n"
"    local dst_size = builtin.uri_unescape(buf, #buf, dst, options.plus)\n"
"    return ffi.string(dst, dst_size)\n"
"end\n"
"\n"
"local function encode_kv(key, values, res, escape_opts)\n"
"    local val = values\n"
"    if type(val) ~= \"table\" then\n"
"        val = { val }\n"
"    end\n"
"    local key_escaped= escape(tostring(key), escape_opts)\n"
"\n"
"    -- { a = {} } --> \"a\"\n"
"    if next(val) == nil then\n"
"        table.insert(res, key_escaped)\n"
"    end\n"
"\n"
"    -- { a = { \"b\" } } --> \"a=c\"\n"
"    for _, v in pairs(val) do\n"
"        local val_escaped = escape(tostring(v), escape_opts)\n"
"        local kv = (\"%s=%s\"):format(key_escaped, val_escaped)\n"
"        table.insert(res, kv)\n"
"    end\n"
"end\n"
"\n"
"-- Encode map to a string and perform escaping of keys and values in\n"
"-- parameters.\n"
"local function params(opts, escape_opts)\n"
"    if opts == nil then\n"
"        return \"\"\n"
"    end\n"
"    if type(opts) ~= \"table\" then\n"
"        error(\"Usage: uri.params(table[, escape_opts])\")\n"
"    end\n"
"    if next(opts) == nil then\n"
"        return \"\"\n"
"    end\n"
"    local res = {}\n"
"    for key, value in pairs(opts) do\n"
"        if type(key) ~= \"string\" and\n"
"           type(key) ~= \"number\" then\n"
"            error(\"uri.params: keys must have a type 'string' or 'number'\")\n"
"        end\n"
"        encode_kv(key, value, res, escape_opts)\n"
"    end\n"
"\n"
"    return table.concat(res, '&')\n"
"end\n"
"\n"
"-- Function allows specify multivalue keys to be added to the query string with\n"
"-- params option.\n"
"-- { key = uri.values(\"param1\", \"param2\") }\n"
"local function values(...)\n"
"    return {...}\n"
"end\n"
"\n"
"return {\n"
"    parse_many = parse_many,\n"
"    parse = parse,\n"
"    format = format,\n"
"    values = values,\n"
"    _internal = {\n"
"        params = params,\n"
"        encode_kv = encode_kv,\n"
"    },\n"
"    escape = escape,\n"
"    unescape = unescape,\n"
"    unreserved = unreserved,\n"
"\n"
"    RFC3986 = RFC3986,\n"
"    PATH = PATH,\n"
"    PATH_PART = PATH_PART,\n"
"    QUERY = QUERY,\n"
"    QUERY_PART = QUERY_PART,\n"
"    FRAGMENT = FRAGMENT,\n"
"    FORM_URLENCODED = FORM_URLENCODED,\n"
"};\n"
""
;
