/*
 * Copyright (C) 2014-2026 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#include <cinttypes> /* PRId64 */
#include <QDir>
#include <QFile>
#include <QThread>
#include <QVariant>

#include "src/datovka_shared/crypto/crypto_wrapped.h"
#include "src/datovka_shared/isds/error.h"
#include "src/datovka_shared/isds/type_description.h"
#include "src/datovka_shared/log/log.h"
#include "src/global.h"
#include "src/io/isds_sessions.h"
#include "src/isds/message_functions.h"
#include "src/isds/services.h"
#include "src/isds/session.h"
#include "src/settings/accounts.h"
#include "src/worker/message_emitter.h"
#include "src/worker/task_archive_document.h"

TaskArchiveIsdsDocument::TaskArchiveIsdsDocument(const AcntIdDb &acntIdDb,
    const QString &transactId, const MsgId &msgId,
    enum MessageDirection direction)
    : QObject(),
    m_result(AID_ERR),
    m_isdsError(),
    m_isdsLongError(),
    m_newExpirationTime(),
    m_acntIdDb(acntIdDb),
    m_transactId(transactId),
    m_msgId(msgId),
    m_direction(direction),
    m_reqQuit(false)
{
	Q_ASSERT(m_acntIdDb.isValid());
	Q_ASSERT(m_msgId.isValid());
}

void TaskArchiveIsdsDocument::run(void)
{
	if (Q_UNLIKELY(!m_acntIdDb.isValid())) {
		Q_ASSERT(0);
		return;
	}

	if (Q_UNLIKELY(!m_msgId.isValid())) {
		Q_ASSERT(0);
		return;
	}

	logDebugLv0NL("Starting archive ISDS document task in thread '%p'",
	    (void *) QThread::currentThreadId());

	/* ### Worker task begin. ### */

	Isds::Session *session =
	    GlobInstcs::isdsSessionsPtr->session(m_acntIdDb.username());
	if (Q_NULLPTR != session) {
		session->setAbortXfer(&m_reqQuit);
		connect(session, SIGNAL(progress(qint64, qint64, qint64, qint64)),
		    this, SLOT(watchProgress(qint64, qint64, qint64, qint64)),
		    Qt::DirectConnection); /* Qt::DirectConnection because it must be executed immediatelly. */
	}

	const AcntId acntId(m_acntIdDb);
	logDebugLv1NL("%s", "-----------------------------------------------");
	logDebugLv1NL("Archiving message '%" PRId64 "' for account '%s'.",
	    UGLY_QINT64_CAST m_msgId.dmId(),
	    GlobInstcs::acntMapPtr->acntData(acntId).accountName().toUtf8().constData());
	logDebugLv1NL("%s", "-----------------------------------------------");

	m_result = archiveMessage(m_acntIdDb, m_msgId, m_direction,
	    m_newExpirationTime, m_isdsError, m_isdsLongError, &m_reqQuit);

	if (AID_SUCCESS == m_result) {
		logDebugLv1NL(
		    "Done archiving message '%" PRId64 "' for account '%s'.",
		    UGLY_QINT64_CAST m_msgId.dmId(),
		    GlobInstcs::acntMapPtr->acntData(acntId).accountName().toUtf8().constData());
	} else {
		logErrorNL("Archiving message '%" PRId64 "' for account '%s' failed.",
		    UGLY_QINT64_CAST m_msgId.dmId(),
		    GlobInstcs::acntMapPtr->acntData(acntId).accountName().toUtf8().constData());
	}

	if (Q_NULLPTR != session) {
		session->setAbortXfer(Q_NULLPTR);
		session->disconnect(Q_NULLPTR, this, Q_NULLPTR);
	}

	/*
	 * Prefering to retrun only long error string as most of the errors
	 * aren't actually errors.
	 * TODO -- libdatovka must return status messaga data as receibed from
	 * server.
	 */
	Q_EMIT GlobInstcs::msgProcEmitterPtr->downloadProgressFinished(
	    AcntId(m_acntIdDb.username(), m_acntIdDb.testing()), m_transactId,
	    m_msgId, m_result,
	    m_isdsLongError.isEmpty() ? (m_isdsError + " " + m_isdsLongError) : m_isdsLongError);

	/* ### Worker task end. ### */

	logDebugLv0NL("Archive ISDS document task finished in thread '%p'",
	    (void *) QThread::currentThreadId());
}

void TaskArchiveIsdsDocument::requestQuit(void)
{
	m_reqQuit = true;
}

void TaskArchiveIsdsDocument::watchProgress(qint64 uploadTotal,
    qint64 uploadCurrent, qint64 downloadTotal, qint64 downloadCurrent)
{
	Q_UNUSED(uploadTotal);
	Q_UNUSED(uploadCurrent);

	static qint64 lastUploadCurrent = 0;
	static qint64 lastDownloadCurrent = 0;
	bool downloadStarted = false;

	if (!downloadStarted) {
		if (uploadCurrent > lastUploadCurrent) {
			lastUploadCurrent = uploadCurrent;

			Q_EMIT GlobInstcs::msgProcEmitterPtr->uploadProgress(
			    AcntId(m_acntIdDb.username(), m_acntIdDb.testing()),
			    m_transactId, uploadTotal, uploadCurrent);
		}

		if (downloadCurrent > lastDownloadCurrent) {
			downloadStarted = true;

			Q_EMIT GlobInstcs::msgProcEmitterPtr->uploadProgressFinished(
			    AcntId(m_acntIdDb.username(), m_acntIdDb.testing()),
			    m_transactId, AID_SUCCESS, QString(),
			    QVariant());
		}
	}

	if (downloadStarted) {
		if (downloadCurrent > lastDownloadCurrent) {
			lastDownloadCurrent = downloadCurrent;

			Q_EMIT GlobInstcs::msgProcEmitterPtr->downloadProgress(
			    AcntId(m_acntIdDb.username(), m_acntIdDb.testing()),
			     m_transactId, m_msgId, downloadTotal, downloadCurrent);
		}
	}
}

enum TaskArchiveIsdsDocument::Result TaskArchiveIsdsDocument::archiveMessage(
    const AcntIdDb &acntIdDb, const MsgId &msgId,
    enum MessageDirection msgDirect, QDateTime &newExpirationTime,
    QString &error, QString &longError, volatile bool *reqQuit)
{
	Isds::Session *session =
	    GlobInstcs::isdsSessionsPtr->session(acntIdDb.username());
	if (Q_UNLIKELY(Q_NULLPTR == session)) {
		logErrorNL("Missing active session for username '%s'.",
		    acntIdDb.username().toUtf8().constData());
		return AID_ERR;
	}

	if (Q_UNLIKELY(GlobInstcs::req_halt
	    || ((Q_NULLPTR != reqQuit) && (*reqQuit)))) {
		error = tr("Requested quit.");
		return AID_ABORTED;
	}

	MessageDbSet *dbSet = acntIdDb.messageDbSet();
	MessageDb *messageDb = dbSet->accessMessageDb(msgId.deliveryTime(), false);
	if (Q_UNLIKELY(Q_NULLPTR == messageDb)) {
		Q_ASSERT(0);
		logErrorNL("Cannot access database to read/store message '%" PRId64 "'.",
		    UGLY_QINT64_CAST msgId.dmId());
		return AID_DB_RD_ERR;
	}

	if (Q_UNLIKELY(GlobInstcs::req_halt
	    || ((Q_NULLPTR != reqQuit) && (*reqQuit)))) {
		error = tr("Requested quit.");
		return AID_ABORTED;
	}

	QByteArray resignedRaw;
	QDate resignedExpDate;
	{
		const QByteArray origRaw = messageDb->getCompleteMessageRaw(msgId.dmId());

		if (Q_UNLIKELY(origRaw.isEmpty())) {
			/* TODO -- log error */
			return AID_DB_RD_ERR;
		}

		if (Q_UNLIKELY(GlobInstcs::req_halt
		    || ((Q_NULLPTR != reqQuit) && (*reqQuit)))) {
			error = tr("Requested quit.");
			return AID_ABORTED;
		}

		Isds::Error err = Isds::Service::archiveISDSDocument(session,
		    origRaw, resignedRaw, resignedExpDate);

		if (Q_UNLIKELY((err.code() != Isds::Type::ERR_SUCCESS)
		        || resignedRaw.isEmpty())) {
			error = Isds::Description::descrError(err.code());
			longError = err.longDescr();
			logErrorNL("Archiving of message returned status %d: '%s' '%s'.",
			    err.code(), error.toUtf8().constData(),
			    longError.toUtf8().constData());
			return AID_ISDS_ERROR;
		}
	}

	if (Q_UNLIKELY(GlobInstcs::req_halt
	    || ((Q_NULLPTR != reqQuit) && (*reqQuit)))) {
		error = tr("Requested quit.");
		return AID_ABORTED;
	}

	const bool storeRawInDb = false;

	Isds::Message resignedMessage = Isds::messageFromData(resignedRaw,
	    Isds::LT_MESSAGE);
	/* Store the message. */
#define IMPLICIT_TRANSACTION true /* Must be disabled when using explicit transactions. */
	Task::storeSignedMessage(IMPLICIT_TRANSACTION, msgDirect, *dbSet,
	    resignedMessage, storeRawInDb);
#undef IMPLICIT_TRANSACTION

	{
		QDateTime expir;
		{
			QDateTime incep;
			if (Q_UNLIKELY(!TstEntries::latestTstSigningCertTimes(
			        resignedRaw, incep, expir))) {
				logErrorNL("Couldn't read timestamp expiration data for message '%" PRId64 "'.",
				    UGLY_QINT64_CAST msgId.dmId());
			}
		}
		newExpirationTime = expir;
	}
	return AID_SUCCESS;
}

TaskArchiveIsdsDocumentFile::TaskArchiveIsdsDocumentFile(
    const AcntIdDb &acntIdDb, const QString &transactId, const QString &srcFile,
    const QString &tgtFile)
    : QObject(),
    m_result(AIDF_ERR),
    m_isdsError(),
    m_isdsLongError(),
    m_newExpirationTime(),
    m_acntIdDb(acntIdDb),
    m_transactId(transactId),
    m_srcFile(srcFile),
    m_tgtFile(tgtFile),
    m_reqQuit(false)
{
	Q_ASSERT(m_acntIdDb.isValid());
	Q_ASSERT(!m_srcFile.isEmpty());
	Q_ASSERT(!m_tgtFile.isEmpty());
}

void TaskArchiveIsdsDocumentFile::run(void)
{
	if (Q_UNLIKELY(!m_acntIdDb.isValid())) {
		Q_ASSERT(0);
		return;
	}

	if (Q_UNLIKELY(m_srcFile.isEmpty())) {
		Q_ASSERT(0);
		return;
	}

	if (Q_UNLIKELY(m_tgtFile.isEmpty())) {
		Q_ASSERT(0);
		return;
	}

	logDebugLv0NL("Starting archive ISDS document file task in thread '%p'",
	    (void *) QThread::currentThreadId());

	/* ### Worker task begin. ### */

	Isds::Session *session =
	    GlobInstcs::isdsSessionsPtr->session(m_acntIdDb.username());
	if (Q_NULLPTR != session) {
		session->setAbortXfer(&m_reqQuit);
		connect(session, SIGNAL(progress(qint64, qint64, qint64, qint64)),
		    this, SLOT(watchProgress(qint64, qint64, qint64, qint64)),
		    Qt::DirectConnection); /* Qt::DirectConnection because it must be executed immediatelly. */
	}

	const AcntId acntId(m_acntIdDb);
	logDebugLv1NL("%s", "-----------------------------------------------");
	logDebugLv1NL("Archiving message file '%s' to '%s' for account '%s'.",
	    m_srcFile.toUtf8().constData(), m_tgtFile.toUtf8().constData(),
	    GlobInstcs::acntMapPtr->acntData(acntId).accountName().toUtf8().constData());
	logDebugLv1NL("%s", "-----------------------------------------------");

	m_result = archiveFile(m_acntIdDb, m_srcFile, m_tgtFile,
	    m_newExpirationTime, m_isdsError, m_isdsLongError, &m_reqQuit);

	if (AIDF_SUCCESS == m_result) {
		logDebugLv1NL(
		    "Done archiving message file '%s' to '%s' for account '%s'.",
		    m_srcFile.toUtf8().constData(), m_tgtFile.toUtf8().constData(),
		    GlobInstcs::acntMapPtr->acntData(acntId).accountName().toUtf8().constData());
	} else {
		logErrorNL("Archiving message '%s' to '%s' for account '%s' failed.",
		    m_srcFile.toUtf8().constData(), m_tgtFile.toUtf8().constData(),
		    GlobInstcs::acntMapPtr->acntData(acntId).accountName().toUtf8().constData());
	}

	if (Q_NULLPTR != session) {
		session->setAbortXfer(Q_NULLPTR);
		session->disconnect(Q_NULLPTR, this, Q_NULLPTR);
	}

	/*
	 * Prefering to retrun only long error string as most of the errors
	 * aren't actually errors.
	 * TODO -- libdatovka must return status messaga data as receibed from
	 * server.
	 */
	Q_EMIT GlobInstcs::msgProcEmitterPtr->downloadProgressFinished(
	    AcntId(m_acntIdDb.username(), m_acntIdDb.testing()), m_transactId,
	    MsgId(), m_result,
	    m_isdsLongError.isEmpty() ? (m_isdsError + " " + m_isdsLongError) : m_isdsLongError);

	/* ### Worker task end. ### */

	logDebugLv0NL("Archive ISDS document file task finished in thread '%p'",
	    (void *) QThread::currentThreadId());
}

void TaskArchiveIsdsDocumentFile::requestQuit(void)
{
	m_reqQuit = true;
}

void TaskArchiveIsdsDocumentFile::watchProgress(qint64 uploadTotal,
    qint64 uploadCurrent, qint64 downloadTotal, qint64 downloadCurrent)
{
	Q_UNUSED(uploadTotal);
	Q_UNUSED(uploadCurrent);

	static qint64 lastUploadCurrent = 0;
	static qint64 lastDownloadCurrent = 0;
	static bool downloadStarted = false;

	if (!downloadStarted) {
		if (uploadCurrent > lastUploadCurrent) {
			lastUploadCurrent = uploadCurrent;

			Q_EMIT GlobInstcs::msgProcEmitterPtr->uploadProgress(
			    AcntId(m_acntIdDb.username(), m_acntIdDb.testing()),
			    m_transactId, uploadTotal, uploadCurrent);
		}

		if (downloadCurrent > lastDownloadCurrent) {
			downloadStarted = true;

			Q_EMIT GlobInstcs::msgProcEmitterPtr->uploadProgressFinished(
			    AcntId(m_acntIdDb.username(), m_acntIdDb.testing()),
			    m_transactId, AIDF_SUCCESS, QString(),
			    QVariant());
		}
	}

	if (downloadStarted) {
		if (downloadCurrent > lastDownloadCurrent) {
			lastDownloadCurrent = downloadCurrent;

			Q_EMIT GlobInstcs::msgProcEmitterPtr->downloadProgress(
			    AcntId(m_acntIdDb.username(), m_acntIdDb.testing()),
			     m_transactId, MsgId(), downloadTotal, downloadCurrent);
		}
	}
}

enum TaskArchiveIsdsDocumentFile::Result TaskArchiveIsdsDocumentFile::archiveFile(
    const AcntIdDb &acntIdDb, const QString &srcFile, const QString &tgtFile,
    QDateTime &newExpirationTime,
    QString &error, QString &longError, volatile bool *reqQuit)
{
	Isds::Session *session =
	    GlobInstcs::isdsSessionsPtr->session(acntIdDb.username());
	if (Q_UNLIKELY(Q_NULLPTR == session)) {
		logErrorNL("Missing active session for username '%s'.",
		    acntIdDb.username().toUtf8().constData());
		return AIDF_ERR;
	}

	if (Q_UNLIKELY(GlobInstcs::req_halt
	    || ((Q_NULLPTR != reqQuit) && (*reqQuit)))) {
		error = tr("Requested quit.");
		return AIDF_ABORTED;
	}

	QByteArray resignedRaw;
	QDate resignedExpDate;
	{
		qint64 fSize = 0;
		const char *fData = NULL;
		QFile fin(srcFile);
		if (Q_UNLIKELY(!fin.open(QIODevice::ReadOnly))) {
			error = tr("Cannot open file '%1' for reading.")
			    .arg(QDir::toNativeSeparators(srcFile));
			return AIDF_FILE_RD_ERR;
		}
		fSize = fin.size();
		fData = (char *)fin.map(0, fSize);

		{
			const QByteArray origRaw = QByteArray::fromRawData(fData, fSize);

			if (Q_UNLIKELY(origRaw.isEmpty())) {
				/* TODO -- log error */
				return AIDF_FILE_RD_ERR;
			}

			if (Q_UNLIKELY(GlobInstcs::req_halt
			    || ((Q_NULLPTR != reqQuit) && (*reqQuit)))) {
				error = tr("Requested quit.");
				return AIDF_ABORTED;
			}

			Isds::Error err = Isds::Service::archiveISDSDocument(session,
			    origRaw, resignedRaw, resignedExpDate);

			if (Q_UNLIKELY((err.code() != Isds::Type::ERR_SUCCESS)
			        || resignedRaw.isEmpty())) {
				error = Isds::Description::descrError(err.code());
				longError = err.longDescr();
				logErrorNL("Archiving of message file returned status %d: '%s' '%s'.",
				    err.code(), error.toUtf8().constData(),
				    longError.toUtf8().constData());
				return AIDF_ISDS_ERROR;
			}
		}

		fin.close(); /* File can be closed but object must remain. */
	}

	if (Q_UNLIKELY(GlobInstcs::req_halt
	    || ((Q_NULLPTR != reqQuit) && (*reqQuit)))) {
		error = tr("Requested quit.");
		return AIDF_ABORTED;
	}

	{
		QFile fout(tgtFile);
		if (Q_UNLIKELY(!fout.open(QIODevice::WriteOnly))) {
			error = tr("Cannot open file '%1' for writing.")
			    .arg(QDir::toNativeSeparators(tgtFile));
			return AIDF_FILE_WR_ERR;
		}

		if (Q_UNLIKELY(fout.write(resignedRaw) != resignedRaw.size())) {
			error = tr("Cannot write complete data into file '%1'.")
			    .arg(QDir::toNativeSeparators(tgtFile));
			return AIDF_FILE_WR_ERR;
		}

		fout.flush();
		fout.close();
	}

	if (Q_UNLIKELY(GlobInstcs::req_halt
	    || ((Q_NULLPTR != reqQuit) && (*reqQuit)))) {
		error = tr("Requested quit.");
		return AIDF_ABORTED;
	}

	{
		QDateTime expir;
		{
			QDateTime incep;
			if (Q_UNLIKELY(!TstEntries::latestTstSigningCertTimes(
			        resignedRaw, incep, expir))) {
				logErrorNL("Couldn't read timestamp expiration data for message in file '%s'.",
				    tgtFile.toUtf8().constData());
			}
		}
		newExpirationTime = expir;
	}

	return AIDF_SUCCESS;
}
