/****************************************************************************
 * sched/clock/clock_dow.c
 *
 *   Copyright (C) 2012 Gregory Nutt. All rights reserved.
 *   Author: Gregory Nutt <gnutt@nuttx.org>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name NuttX nor the names of its contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

/****************************************************************************
 * Included Files
 ****************************************************************************/

#include <nuttx/config.h>

#include <stdint.h>

#include <nuttx/clock.h>

#include "clock/clock.h"

/****************************************************************************
 * Private Data
 ****************************************************************************/

/* 23 * (month + 1) / 9, month = 0..11 */

static const uint8_t g_lookup[12] =
{
  2, 5, 7, 10, 12, 15, 17, 20, 23, 25, 28, 30
};

/****************************************************************************
 * Public Functions
 ****************************************************************************/

/****************************************************************************
 * Name: clock_dow
 *
 * Description:
 *   Calculate the day of week (DOW) from they year month and day.  Based on
 *   an algorithm pubished in 1990 by Michael Keith and Tom Craver with some
 *   tweaks to handle months in the range 0-11.
 *
 * Input Parameters:
 *   year  - year (e.g., 1988)
 *   month - 0 through 11
 *   day   - 1 through 31
 *
 * Returned Value:
 *   The day of the week as days since Sunday: 0 = Sunday, 1 = Monday, etc.
 *
 * Assumptions:
 *
 ****************************************************************************/

int clock_dow(int year, int month, int day)
{
  day += month < 2 ? year-- : year - 2;
  return ((int)g_lookup[month] + day + 4 + year / 4 - year / 100 + year / 400) % 7;
}
