/* Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 *
 */


package carstore;

import javax.faces.component.UIComponent;
import javax.faces.component.ValueHolder;
import javax.faces.context.FacesContext;
import javax.faces.convert.Converter;

import java.util.Enumeration;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;


/**
 * <p>A helper class that customizes a CarBean for a set of options
 * in a package.</p>
 * <p/>
 * <p>This class reads its settings from a Properties file</p>
 */

public class CarCustomizer {

    private static final Logger LOGGER = Logger.getLogger("carstore");

    //
    // Relationship Instance Variables
    //

    private ResourceBundle bundle = null;


    public CarCustomizer() {
        this.init(CarStore.DEFAULT_PACKAGE_PROPERTIES);
    }


    public CarCustomizer(String bundleName) {
        this.init(bundleName);
    }


    private void init(String bundleName) {

        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.fine("Loading bundle: " + bundleName + ".");
        }
        bundle = ResourceBundle.getBundle(bundleName);
    }


    private String buttonStyle = null;


    public String getButtonStyle() {
        return buttonStyle;
    }


    public void setButtonStyle(String newButtonStyle) {
        buttonStyle = newButtonStyle;
    }


    public void customizeCar(CarBean toCustomize) {

        FacesContext context = FacesContext.getCurrentInstance();
        String disabledStr;
        String curSetting;
        UIComponent component;
        Converter converter;
        Object valueToSet;

        for (Enumeration keys = bundle.getKeys(); keys.hasMoreElements();) {

            String key = (String) keys.nextElement();
            // skip null and secondary keys.
            if (key == null || -1 != key.indexOf('_')) {
                continue;
            }
            // skip null values
            if (null == (curSetting = bundle.getString(key))) {
                continue;
            }

            // skip null components
            if (null ==
                (component =
                      (UIComponent) toCustomize.getComponents().get(key))) {
                continue;
            }

            // handle the disabled setting, if necessary
            Boolean disabled = null;
            try {
                if (null !=
                    (disabledStr = bundle.getString(key + "_disabled"))) {
                    disabled = Boolean.valueOf(disabledStr);
                }
            } catch (Throwable e) {
                // do nothing
            }
            if (null != disabled) {
                component.getAttributes().put("disabled", disabled);
            }

            // set the value
            // If the component can and does have a converter
            if (component instanceof ValueHolder &&
                (null != (converter =
                      ((ValueHolder) component).getConverter()))) {
                valueToSet = converter.getAsObject(context, component,
                                                   curSetting);
            } else {
                valueToSet = curSetting;
            }

            if (component instanceof ValueHolder) {
                ((ValueHolder) component).setValue(valueToSet);
            }
        }
    }
}
	
	
	
