/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.kyuubi.server.http.util

import java.security.SecureRandom
import java.util
import java.util.StringTokenizer

import scala.collection.mutable

import org.apache.kyuubi.Logging

object HttpAuthUtils extends Logging {
  val WWW_AUTHENTICATE = "WWW-Authenticate"
  val AUTHORIZATION = "Authorization"
  val BASIC = "Basic"
  val NEGOTIATE = "Negotiate"

  private val COOKIE_ATTR_SEPARATOR = "&"
  private val COOKIE_CLIENT_USER_NAME = "cu"
  private val COOKIE_CLIENT_RAND_NUMBER = "rn"
  private val COOKIE_KEY_VALUE_SEPARATOR = "="
  private val COOKIE_ATTRIBUTES = new util.HashSet[String](
    util.Arrays.asList(COOKIE_CLIENT_USER_NAME, COOKIE_CLIENT_RAND_NUMBER))

  /**
   * Creates and returns a HS2 cookie token.
   *
   * @param clientUserName Client User name.
   * @return An unsigned cookie token generated from input parameters.
   *         The final cookie generated is of the following format :
   *         cu=<username>&rn=<randomNumber>&s=<cookieSignature>
   */
  def createCookieToken(clientUserName: String): String = {
    val sb = new mutable.StringBuilder
    sb.append(COOKIE_CLIENT_USER_NAME).append(COOKIE_KEY_VALUE_SEPARATOR)
      .append(clientUserName).append(COOKIE_ATTR_SEPARATOR).append(COOKIE_CLIENT_RAND_NUMBER)
      .append(COOKIE_KEY_VALUE_SEPARATOR).append((new SecureRandom).nextLong)
    sb.toString
  }

  /**
   * Parses a cookie token to retrieve client user name.
   *
   * @param tokenStr Token String.
   * @return A valid user name if input is of valid format, else returns null.
   */
  def getUserNameFromCookieToken(tokenStr: String): String = {
    val map = splitCookieToken(tokenStr)
    if (!(map.keySet == COOKIE_ATTRIBUTES)) {
      error("Invalid token with missing attributes " + tokenStr)
      return null
    }
    map.get(COOKIE_CLIENT_USER_NAME)
  }

  /**
   * Splits the cookie token into attributes pairs.
   *
   * @param tokenStr input token.
   * @return a map with the attribute pairs of the token if the input is valid.
   *         Else, returns null.
   */
  private def splitCookieToken(tokenStr: String): util.Map[String, String] = {
    val map = new util.HashMap[String, String]
    val st = new StringTokenizer(tokenStr, COOKIE_ATTR_SEPARATOR)
    while ({
      st.hasMoreTokens
    }) {
      val part = st.nextToken
      val separator = part.indexOf(COOKIE_KEY_VALUE_SEPARATOR)
      if (separator == -1) {
        error("Invalid token string " + tokenStr)
        return null
      }
      val key = part.substring(0, separator)
      val value = part.substring(separator + 1)
      map.put(key, value)
    }
    map
  }
}
