/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.apache.hop.database.cassandra.util;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertNull;

import org.apache.hop.databases.cassandra.util.CqlUtils;
import org.apache.hop.databases.cassandra.util.Selector;
import org.junit.jupiter.api.Test;

class CqlUtilsTest {

  /** */
  private static final String ALIAS_IS_INCORRECT = "Alias is incorrect:";

  /** */
  private static final String COLUMN_NAME_IS_INCORRECT = "Column Name is incorrect:";

  private String cqlExpression;
  private String expected;
  private String selectorExpression;
  private boolean isCql3;
  private Selector selector;
  private Selector expectedSelector;
  private String result;
  private Selector[] selectors;
  private Selector[] expectedSelectors;

  @Test
  void testCleanQoutesMixed() {
    cqlExpression = "one_more as \"\"ALIAS\", field(a, b) as '\"aaaaa'";
    expected = "one_more as ALIAS, field(a, b) as aaaaa";
    result = CqlUtils.cleanQuotes(cqlExpression);
    assertEquals(expected, result);
  }

  @Test
  void testCleanDoubleQoutes_IfAtTheBeginAndEnd() {
    cqlExpression = "\"Test\"";
    expected = "Test";
    result = CqlUtils.cleanQuotes(cqlExpression);
    assertEquals(expected, result);
  }

  @Test
  void testCleanQoutes_IfAtTheBeginAndEnd() {
    cqlExpression = "'Test'";
    expected = "Test";
    result = CqlUtils.cleanQuotes(cqlExpression);
    assertEquals(expected, result);
  }

  @Test
  void testCleanQoutes_IfInputNull() {
    cqlExpression = null;
    result = CqlUtils.cleanQuotes(cqlExpression);
    assertNull(result);
  }

  @Test
  void testCleanMultipleWhitespaces() {
    cqlExpression = "     Test Test  Test   Test      Test    ";
    expected = "Test Test Test Test Test";
    result = CqlUtils.clean(cqlExpression);
    assertEquals(expected, result);
  }

  @Test
  void testCleanUnnecessaryWhitespacesInFunctions() {
    cqlExpression = "function      (     arg1     ,      arg2     ,    arg3    )   as alias";
    expected = "function(arg1, arg2, arg3) as alias";
    result = CqlUtils.clean(cqlExpression);
    assertEquals(expected, result);
  }

  @Test
  void testAddWhitespaceAfterComma() {
    cqlExpression = "selector1,selector2,selector3,function(a,b)";
    expected = "selector1, selector2, selector3, function(a, b)";
    result = CqlUtils.clean(cqlExpression);
    assertEquals(expected, result);
  }

  @Test
  void testClean_IfInputNull() {
    cqlExpression = null;
    result = CqlUtils.clean(cqlExpression);
    assertNull(result);
  }

  @Test
  void testGetSelectExpression_SelectClauseMixedWithUpperCase() {
    cqlExpression =
        "SELECT FIRST 25 DISTINCT selector1, selector2, selector3, f(a,b), selector4 as alias4 from cf where";
    expected = "selector1, selector2, selector3, f(a, b), selector4 as alias4";
    result = CqlUtils.getSelectExpression(cqlExpression);
    assertEquals(expected, result);
  }

  @Test
  void testGetSelectExpression_SelectClauseWithFirstAndCorrectNumber() {
    cqlExpression =
        "select first    5054678   selector1, selector2, selector3, f(a,b), selector4 as alias4 from cf where";
    expected = "selector1, selector2, selector3, f(a, b), selector4 as alias4";
    result = CqlUtils.getSelectExpression(cqlExpression);
    assertEquals(expected, result);
  }

  @Test
  void testGetSelectExpression_SelectClauseWithFirstAndWithoutNumber() {
    cqlExpression =
        "select first selector1, selector2, selector3, f(a,b), selector4 as alias4 from cf where";
    expected = "selector1, selector2, selector3, f(a, b), selector4 as alias4";
    result = CqlUtils.getSelectExpression(cqlExpression);
    assertEquals(expected, result);
  }

  @Test
  void testGetSelectExpression_SelectClauseWithDistinct() {
    cqlExpression =
        "select distinct selector1, selector2, selector3, f(a,b), selector4 as alias4 from cf where";
    expected = "selector1, selector2, selector3, f(a, b), selector4 as alias4";
    result = CqlUtils.getSelectExpression(cqlExpression);
    assertEquals(expected, result);
  }

  @Test
  void testGetSelectExpression_IfInputNull() {
    cqlExpression = null;
    result = CqlUtils.getSelectExpression(cqlExpression);
    assertNull(result);
  }

  @Test
  void testGetSelectExpression_IfInputIsNotSelectClause() {
    cqlExpression = "test input, not select clause";
    result = CqlUtils.getSelectExpression(cqlExpression);
    assertNull(result);
  }

  @Test
  void testSelectorForCQL3ColumnWithouAlias_NameShouldBeInLowerCase() {
    selectorExpression = "User";
    expectedSelector = new Selector("user");
    isCql3 = true;
    selector = CqlUtils.buildSelector(selectorExpression, isCql3);
    assertEquals(
        expectedSelector.getColumnName(), selector.getColumnName(), COLUMN_NAME_IS_INCORRECT);
    assertNull(selector.getAlias());
    assertNull(selector.getFunction());
    assertFalse(selector.isFunction());
  }

  @Test
  void testSelectorForCQL3ColumnWithouAlias_NameShouldBeInOriginCase() {
    selectorExpression = "\"User\"";
    expectedSelector = new Selector("User");
    isCql3 = true;
    selector = CqlUtils.buildSelector(selectorExpression, isCql3);
    assertEquals(
        expectedSelector.getColumnName(), selector.getColumnName(), COLUMN_NAME_IS_INCORRECT);
    assertNull(selector.getAlias());
    assertNull(selector.getFunction());
    assertFalse(selector.isFunction());
  }

  @Test
  void testSelectorForCQLColumnWithouAlias_NameShouldNotBeInLowerCase() {
    selectorExpression = "User";
    expectedSelector = new Selector("User");
    isCql3 = false;
    selector = CqlUtils.buildSelector(selectorExpression, isCql3);
    assertEquals(
        expectedSelector.getColumnName(), selector.getColumnName(), COLUMN_NAME_IS_INCORRECT);
    assertNull(selector.getAlias());
    assertNull(selector.getFunction());
    assertFalse(selector.isFunction());
  }

  @Test
  void testSelectorForCQL3ColumnWithAlias_AliasShouldBeInLowerCase() {
    selectorExpression = "User as Alias";
    expectedSelector = new Selector("user", "alias");
    isCql3 = true;
    selector = CqlUtils.buildSelector(selectorExpression, isCql3);
    assertEquals(
        expectedSelector.getColumnName(), selector.getColumnName(), COLUMN_NAME_IS_INCORRECT);
    assertEquals(expectedSelector.getAlias(), selector.getAlias());
    assertNull(selector.getFunction());
    assertFalse(selector.isFunction());
  }

  @Test
  void testSelectorForCQL3ColumnWithAlias_AliasShouldBeInOriginCase() {
    selectorExpression = "User as \"Alias\"";
    expectedSelector = new Selector("user", "Alias");
    isCql3 = true;
    selector = CqlUtils.buildSelector(selectorExpression, isCql3);
    assertEquals(
        expectedSelector.getColumnName(), selector.getColumnName(), COLUMN_NAME_IS_INCORRECT);
    assertEquals(expectedSelector.getAlias(), selector.getAlias());
    assertNull(selector.getFunction());
    assertFalse(selector.isFunction());
  }

  @Test
  void testSelectorForCQLColumnWithAlias_AliasShouldNotBeInLowerCase() {
    selectorExpression = "User as Alias";
    expectedSelector = new Selector("User", "Alias");
    isCql3 = false;
    selector = CqlUtils.buildSelector(selectorExpression, isCql3);
    assertEquals(
        expectedSelector.getColumnName(), selector.getColumnName(), COLUMN_NAME_IS_INCORRECT);
    assertEquals(expectedSelector.getAlias(), selector.getAlias());
    assertNull(selector.getFunction());
    assertFalse(selector.isFunction());
  }

  @Test
  void testSelectorForCQL3FunctionWithAlias() {
    selectorExpression = "token(a, b) as \"Alias\"";
    expectedSelector = new Selector("token(a, b)", "Alias", "TOKEN");
    isCql3 = true;
    selector = CqlUtils.buildSelector(selectorExpression, isCql3);
    assertEquals(
        expectedSelector.getColumnName(), selector.getColumnName(), COLUMN_NAME_IS_INCORRECT);
    assertEquals(expectedSelector.getAlias(), selector.getAlias());
    assertEquals(expectedSelector.getFunction(), selector.getFunction());
    assertEquals(expectedSelector.isFunction(), selector.isFunction());
  }

  @Test
  void testSelectorForCQL3FunctionWithoutAlias() {
    selectorExpression = "token(a, b)";
    expectedSelector = new Selector("token(a, b)", null, "TOKEN");
    isCql3 = true;
    selector = CqlUtils.buildSelector(selectorExpression, isCql3);
    assertEquals(
        expectedSelector.getColumnName(), selector.getColumnName(), COLUMN_NAME_IS_INCORRECT);
    assertNull(selector.getAlias());
    assertEquals(expectedSelector.getFunction(), selector.getFunction());
    assertEquals(expectedSelector.isFunction(), selector.isFunction());
  }

  @Test
  void testSelectorForCQL3FunctionName_ShouldBeInOriginCase() {
    selectorExpression = "TOKEN(a, b)";
    expectedSelector = new Selector("TOKEN(a, b)", null, "TOKEN");
    isCql3 = false;
    selector = CqlUtils.buildSelector(selectorExpression, isCql3);
    assertEquals(
        expectedSelector.getColumnName(), selector.getColumnName(), COLUMN_NAME_IS_INCORRECT);
    assertNull(selector.getAlias());
    assertEquals(expectedSelector.getFunction(), selector.getFunction());
    assertEquals(expectedSelector.isFunction(), selector.isFunction());
  }

  @Test
  void testOneFunctionWithoutAliasInSelectorList_CQL3TurnedOn() {
    cqlExpression = "token(a,b)";
    expectedSelector = new Selector("token(a, b)", null, "TOKEN");
    isCql3 = true;

    selectors = CqlUtils.getColumnsInSelect(cqlExpression, isCql3);

    assertNotNull(selectors);
    assertEquals(1, selectors.length);
    // check selector
    assertEquals(
        expectedSelector.getColumnName(), selectors[0].getColumnName(), COLUMN_NAME_IS_INCORRECT);
    assertNull(selectors[0].getAlias());
    assertEquals(expectedSelector.getFunction(), selectors[0].getFunction());
    assertEquals(expectedSelector.isFunction(), selectors[0].isFunction());
  }

  @Test
  void testOneFunctionWithoutAliasInSelectorList_CQL3TurnedOFF() {
    cqlExpression = "token(a,b)";
    expectedSelector = new Selector("token(a, b)", null, "TOKEN");
    isCql3 = false;

    selectors = CqlUtils.getColumnsInSelect(cqlExpression, isCql3);

    assertNotNull(selectors);
    assertEquals(1, selectors.length);
    // check selector
    assertEquals(
        expectedSelector.getColumnName(), selectors[0].getColumnName(), COLUMN_NAME_IS_INCORRECT);
    assertNull(selectors[0].getAlias());
    assertEquals(expectedSelector.getFunction(), selectors[0].getFunction());
    assertEquals(expectedSelector.isFunction(), selectors[0].isFunction());
  }

  @Test
  void testOneColumnWithoutAliasInSelectorList_CQL3TurnedOn() {
    cqlExpression = "Column";
    expectedSelector = new Selector("column");
    isCql3 = true;

    selectors = CqlUtils.getColumnsInSelect(cqlExpression, isCql3);

    assertNotNull(selectors);
    assertEquals(1, selectors.length);
    // check selector
    assertEquals(
        expectedSelector.getColumnName(), selectors[0].getColumnName(), COLUMN_NAME_IS_INCORRECT);
    assertNull(selectors[0].getAlias());
    assertNull(selectors[0].getFunction());
    assertFalse(selectors[0].isFunction());
  }

  @Test
  void testOneColumnWithoutAliasInSelectorList_CQL3TurnedOFF() {
    cqlExpression = "Column";
    expectedSelector = new Selector("Column");
    isCql3 = false;

    selectors = CqlUtils.getColumnsInSelect(cqlExpression, isCql3);

    assertNotNull(selectors);
    assertEquals(1, selectors.length);
    // check selector
    assertEquals(
        expectedSelector.getColumnName(), selectors[0].getColumnName(), COLUMN_NAME_IS_INCORRECT);
    assertNull(selectors[0].getAlias());
    assertNull(selectors[0].getFunction());
    assertFalse(selectors[0].isFunction());
  }

  @Test
  void testMixedColumnsAndFunctionsWithoutAliasInSelectorList_CQL3TurnedOn() {
    cqlExpression = "\"Column\", token(id,cust_id), id, cust_id, dateOf(time), User_name";
    expectedSelectors =
        new Selector[] {
          new Selector("Column"),
          new Selector("token(id, cust_id)", null, "TOKEN"),
          new Selector("id"),
          new Selector("cust_id"),
          new Selector("dateOf(time)", null, "DATEOF"),
          new Selector("user_name")
        };
    isCql3 = true;

    selectors = CqlUtils.getColumnsInSelect(cqlExpression, isCql3);

    verifySelectors(expectedSelectors, selectors);
  }

  @Test
  void testMixedColumnsAndFunctionsWithoutAliasInSelectorList_CQL3TurnedOFF() {
    cqlExpression = "'Column', token(id,cust_id), id, cust_id, dateOf(time), User_name";
    expectedSelectors =
        new Selector[] {
          new Selector("Column"),
          new Selector("token(id, cust_id)", null, "TOKEN"),
          new Selector("id"),
          new Selector("cust_id"),
          new Selector("dateOf(time)", null, "DATEOF"),
          new Selector("User_name")
        };
    isCql3 = false;

    selectors = CqlUtils.getColumnsInSelect(cqlExpression, isCql3);

    verifySelectors(expectedSelectors, selectors);
  }

  @Test
  void testMixedColumnsAndFunctionsWithAliasInSelectorList_CQL3TurnedOn() {
    cqlExpression =
        "\"Column\" as alias, token(id,cust_id) as \"Alias For Token\", id, cust_id as Customer, dateOf(time), User_name";
    expectedSelectors =
        new Selector[] {
          new Selector("Column", "alias"),
          new Selector("token(id, cust_id)", "Alias For Token", "TOKEN"),
          new Selector("id"),
          new Selector("cust_id", "customer"),
          new Selector("dateOf(time)", null, "DATEOF"),
          new Selector("user_name")
        };
    isCql3 = true;

    selectors = CqlUtils.getColumnsInSelect(cqlExpression, isCql3);

    verifySelectors(expectedSelectors, selectors);
  }

  @Test
  void testSpecificNameForCountFunctionWithArgumentAsterisk() {
    cqlExpression = "count(*)";
    expectedSelectors = new Selector[] {new Selector("COUNT", null, "COUNT")};
    isCql3 = true;

    selectors = CqlUtils.getColumnsInSelect(cqlExpression, isCql3);

    verifySelectors(expectedSelectors, selectors);
  }

  @Test
  void testSpecificNameForCountFunctionWithArgumentOne() {
    cqlExpression = "count(1)";
    expectedSelectors = new Selector[] {new Selector("COUNT", null, "COUNT")};
    isCql3 = true;

    selectors = CqlUtils.getColumnsInSelect(cqlExpression, isCql3);

    verifySelectors(expectedSelectors, selectors);
  }

  @Test
  void testColumnsInSelect_IfSelectExpressionNull() {
    cqlExpression = null;
    isCql3 = true;

    selectors = CqlUtils.getColumnsInSelect(cqlExpression, isCql3);
    assertNull(selectors);
  }

  @Test
  void testColumnsInSelect_IfSelectExpressionEmpty() {
    cqlExpression = "";
    isCql3 = true;

    selectors = CqlUtils.getColumnsInSelect(cqlExpression, isCql3);
    assertNull(selectors);
  }

  private void verifySelectors(Selector[] expected, Selector[] actual) {
    assertNotNull(actual);
    assertEquals(expected.length, actual.length);
    // check selectors
    for (int i = 0; i < expected.length; i++) {
      assertEquals(
          expected[i].getColumnName(), actual[i].getColumnName(), COLUMN_NAME_IS_INCORRECT);
      if (expected[i].getAlias() != null) {
        assertEquals(expected[i].getAlias(), actual[i].getAlias(), ALIAS_IS_INCORRECT);
      } else {
        assertNull(actual[i].getAlias());
      }
      assertEquals(expected[i].isFunction(), actual[i].isFunction());
      if (expected[i].isFunction()) {
        assertEquals(expected[i].getFunction(), actual[i].getFunction());
      } else {
        assertNull(actual[i].getFunction());
      }
    }
  }
}
